

/*
   *  Object %name    : %
   *  State           :  %state%
   *  Creation date   :  Wed Jan 12 12:16:55 2005
   *  Last modified   :  %modify_time%
   */
  /** @file
   *  \brief A brief description of this module
   *
   *  \version CRYS_ATP_TST_RSA_Acceptance.c#1:csrc:1
   *  \author doronp
   *  \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
   *           All Rights reserved
   */

/***** Include Files ******/ 
#include "CRYS.h"
 
#include "CRYS_ATP_TST_Defines.h"
#include "CRYS_ATP_TST_UTIL_Funcs.h"
#include "CRYS_ATP_TST_RSA_Acceptance.h"
#include "CRYS_ATP_TST_RSA_Acceptance_test_vectors.h"

#include "DX_VOS_Mem.h"
#include "math.h"
#include <stdio.h>
#if TST_USE_RSA_DATA/*In case no test vectors are selected - code disabled*/
/******** Defines ********/
/******** Enums **********/
/******** Typedefs *********/
/******** Global Data ********/
static  CRYS_RSAUserPrivKey_t       TST_PrivateKey;
static  CRYS_RSAUserPubKey_t        TST_PublicKey;
static  CRYS_RSAPrimeData_t         TST_PrimeData;
static  CRYS_RSAKGData_t            TST_KeyGenData;
static  CRYS_RSAPrivUserContext_t   TST_ContextPrivate; 
static  CRYS_RSAPubUserContext_t    TST_ContextPublic;
/***** Private function prototype ******/
/******** Private Functions ******/
/******** Public Functions *****/
/*******************
 * Function Name:                                                           
 *  CRYS_ATP_TST_RSA_Acceptance_PRIME_EncDec        
 * 
 * Inputs:
 *  None
 *
 * Outputs:
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for RSA primitive encrypt and decrypt                    
 *  This function Encrypts and Decrypts known vectors using RSA primitives
 *  Vector/Keys sizes used (in bits) - 512/1024/2048                                                        
 *  This also outputs the results to an output buffer or the same input buffer
 * Algorithm:       
 *  1. Build the Private Key from external exponent and modulus
 *  2. Build the Public Key from External exponent and modulus
 *  3. Encrypt/Decrypt a known vector
 *  4. Compare the outputted result to the expected one                  
 *
 **************************/                                
CRYSError_t CRYS_ATP_TST_RSA_Acceptance_PRIME_EncDec ( void )
{
    /*** TST Variables Definitions ****/
    int         TST_index,TST_CRT_Mode;
    CRYSError_t TST_Status=0;
    DxUint16_t  TST_GetModulusSize;
    char        TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_RSA_Acceptance_PRIME_EncDec";   
    
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"*********************** \n"));
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_RSA_Acceptance_PRIME_EncDec Tests   ******* \n"));      
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"*********************** \n\n"));
    
    /****************/
    /***** Start of test ********/
    /****************/
    for (TST_CRT_Mode = 0 ; TST_CRT_Mode < 2 ; TST_CRT_Mode++)/* TST_CRT_Mode = 0 no CRT // TST_CRT_Mode = 1 CRT mode*/
    {
        for (TST_index =0 ; TST_index < TST_NumberOf_RSA_AcceptanceTests ; TST_index++) /*To run on all test vectors*/
        {
            if (!((TST_CRT_Mode == TST_CRT_MODE) && 
                 (TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize == TST_RSA_512_KEY_SIZE_IN_BYTES)))
            { 
                /* output buffer != input buffer    */
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"************************ \n"));
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," %s[test 1.%d]\n",TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Name,TST_index));  
                if (TST_CRT_Mode == TST_NO_CRT)
                    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," Non CRT mode  \n"));
                else
                    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," CRT mode  \n"));
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"************************ \n"));

                /* Build the public Key */
                /*----------------------*/
                TST_Status = CRYS_RSA_Build_PubKey(&TST_PublicKey,
                                                    TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_PublicExponent_E,
                                                    TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_PubExponentSize,
                                                    TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Modulus_N,
                                                    TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize);
                /* Get the E size */
                /*----------------*/
                TST_Status = CRYS_RSA_Get_ModSizeFromPubKey(&TST_PublicKey,
                                                    &TST_GetModulusSize);
                
                TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Get_ModSizeFromPubKey",
                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Name,
                                                        TST_MasterTestName);


                /* Check that received E size is correct */
                /*---------------------------------------*/
               if(TST_GetModulusSize != TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize)
                {
                  TST_Status = TESTFAIL;
                }
               else
                {
                  TST_Status = TESTPASS;
                }

                TST_ATP_CHECK_ERROR_RETURN(TST_Status,"Compare - TST_GetModulusSize",
                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Name,
                                                        TST_MasterTestName);
            

                                                
                TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Build_PubKey",
                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Name,
                                                        TST_MasterTestName);
  
                            
                /* Build the Private Key */
                /*-----------------------*/ 
                if (TST_CRT_Mode == TST_NO_CRT)
                {
                    TST_Status = CRYS_RSA_Build_PrivKey(&TST_PrivateKey,
                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_PrivetExponent_D,
                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize,
                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_PublicExponent_E,
                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_PubExponentSize,                                                   
                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Modulus_N,
                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize);                                  
                    
                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Build_PrivKey",
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Name,
                                                            TST_MasterTestName);

                }
                else
                {
                    TST_Status = CRYS_RSA_Build_PrivKeyCRT(&TST_PrivateKey,
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_P,
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_DPSize,
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Q,                                                                                                 
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_DPSize,
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_dP,                                                                                                
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_DPSize, 
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_dQ,                                                                                                
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_DPSize,                                                                                                
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Qinv,
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_DPSize);
                    
                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Build_PrivKeyCRT",
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Name,
                                                            TST_MasterTestName);
 
                }
                
                /* Encrypt the data using the primes */
                /* (Public Key)                      */
                /*-----------------------------------*/ 
                TST_ATP_StartPerformanceTimer(&TST_Timer);

                TST_Status = CRYS_RSA_PRIM_Encrypt(&TST_PublicKey,
                                                      &TST_PrimeData,
                                                      TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_input_data,
                                                      TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize,
                                                      TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_output_data);   
                 
                TST_ATP_GetPerformanceTimer(&TST_Timer);
                TST_ATP_CalcPerformance(TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize);

                TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_PRIM_Encrypt",
                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Name,
                                                        TST_MasterTestName);

                
                /*Print the outputted and then the expected result before mem compare */
                ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                        "\nData outputted by engine [output buffer]-- \n",
                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_output_data,
                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize);
                                            
                ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                        "\nData Expected by engine [expected buffer - encrypt]-- \n",
                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Expected_output_data,
                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize);                                                          
                
                /* Compare the Encrypted data to the expected */
                /*--------------------------------------------*/
                TST_Status = (CRYSError_t)DX_VOS_MemCmp(TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_output_data ,
                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Expected_output_data,
                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize);                  

               TST_ATP_CHECK_ERROR_RETURN(TST_Status,"DX_VOS_MemCmp - Encrypt",
                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Name,
                                                        TST_MasterTestName);

                
                /* Decrypt the data using the primes */
                /* (Private Key)                     */
                /*-----------------------------------*/ 
                TST_ATP_StartPerformanceTimer(&TST_Timer);

                TST_Status = CRYS_RSA_PRIM_Decrypt(&TST_PrivateKey,
                                                    &TST_PrimeData,
                                                    TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_output_data,
                                                    TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize,
                                                    TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_output_data);

                TST_ATP_GetPerformanceTimer(&TST_Timer);
                TST_ATP_CalcPerformance(TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize);
                                                  
                TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_PRIM_Decrypt",
                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Name,
                                                        TST_MasterTestName);
 
                                                          
                /*Print the outputted and then the expected result before mem compare */
                ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                        "\nData outputted by engine [output buffer]-- \n",
                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_output_data,
                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize);
                                           
                ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                        "\nData Expected by engine [Input buffer - Decrypt]-- \n",
                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_input_data,
                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize);                                                                                
                                             
                TST_Status = (CRYSError_t)DX_VOS_MemCmp(TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_output_data ,
                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_input_data,
                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize);  
                 
                TST_ATP_CHECK_ERROR_RETURN(TST_Status,"DX_VOS_MemCmp - Decrypt",
                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Name,
                                                        TST_MasterTestName);

            }/* End of CRT constraint*/
        }/*End of TST_Index loop*/
    }/*End of TST_CRT_Mode loop*/
  return TST_Status; 
   
EXIT_ON_ERROR:
  {
   return DX_FAILURE;
  }
}/*End of CRYS_ATP_TST_RSA_Acceptance_PRIME_EncDec*/
/*****************************************************/
/*TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************/
/*--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************/
/*TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************/
/*--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************/
/*****************************************************/
/***************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_RSA_Acceptance_PRIME_SignVerify        
 * 
 * Inputs:
 *  None
 *
 * Outputs:
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for RSA primitive Sign & Verify                  
 *  This function signs and verifies known vectors using RSA primitives
 *  Vector/Keys sizes used (in bits) - 512/1024/2048                                                        
 *  This also outputs the results to an output buffer or the same input buffer
 * Algorithm:       
 *  1. Build the Private Key from external exponent and modulus
 *  2. Build the Public Key from External exponent and modulus
 *  3. sign/verify a known vector
 *  4. Compare the outputted result to the expected one                  
 *
 **************************/                                
CRYSError_t CRYS_ATP_TST_RSA_Acceptance_PRIME_SignVerify ( void )
{
    /*** TST Variables Definitions ****/
    int         TST_index,TST_CRT_Mode;
    CRYSError_t TST_Status=0;
    DxUint16_t    TST_GetModulusSize;
    char        TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_RSA_Acceptance_PRIME_SignVerify";   

    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"*********************** \n"));
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_RSA_Acceptance_PRIME_SignVerify Tests   ******* \n"));      
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"*********************** \n \n"));
    
    /****************/
    /***** Start of test ********/
    /****************/
    for (TST_CRT_Mode = 0 ; TST_CRT_Mode < 2 ; TST_CRT_Mode++)/* TST_CRT_Mode = 0 no CRT // TST_CRT_Mode = 1 CRT mode*/
    {
        for (TST_index =0 ; TST_index < TST_NumberOf_RSA_AcceptanceTests ; TST_index++) /*To run on all test vectors*/
        {  
            if (!((TST_CRT_Mode == TST_CRT_MODE) && 
                 (TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize == TST_RSA_512_KEY_SIZE_IN_BYTES)))
            { 
                /* output buffer != input buffer    */
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"************************ \n"));
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," %s[test 2.%d]\n",TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Name,TST_index));  
                if (TST_CRT_Mode == TST_NO_CRT)
                    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," Non CRT mode  \n"));
                else
                    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," CRT mode  \n"));
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"************************ \n"));     
                
                /* Build the public Key */
                /*----------------------*/
                TST_Status = CRYS_RSA_Build_PubKey(&TST_PublicKey,
                                                    TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_PublicExponent_E,
                                                    TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_PubExponentSize,
                                                    TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Modulus_N,
                                                    TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize);
                                                
                TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Build_PubKey",
                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Name,
                                                        TST_MasterTestName);
   

                /* Get the E size */
                /*----------------*/
                TST_Status = CRYS_RSA_Get_ModSizeFromPubKey(&TST_PublicKey,
                                                    &TST_GetModulusSize);
                
                TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Get_ModSizeFromPubKey",
                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Name,
                                                        TST_MasterTestName);


                /* Check that received E size is correct */
                /*---------------------------------------*/
               if(TST_GetModulusSize != TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize)
                {
                  TST_Status =TESTFAIL;
                }
                else
                {
                  TST_Status =TESTPASS;
                }


                TST_ATP_CHECK_ERROR_RETURN(TST_Status,"Compare - TST_GetModulusSize",
                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Name,
                                                        TST_MasterTestName);
             

                /* Build the Private Key */
                /*-----------------------*/ 
                if (TST_CRT_Mode == TST_NO_CRT)
                {
                    TST_Status = CRYS_RSA_Build_PrivKey(&TST_PrivateKey,
                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_PrivetExponent_D,
                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize,
                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_PublicExponent_E,
                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_PubExponentSize,                                                   
                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Modulus_N,
                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize);                                  
                    
                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Build_PrivKey",
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Name,
                                                            TST_MasterTestName);

                }
                else
                {
                    TST_Status = CRYS_RSA_Build_PrivKeyCRT(&TST_PrivateKey,
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_P,
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_DPSize,
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Q,                                                                                                 
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_DPSize,
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_dP,                                                                                                
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_DPSize, 
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_dQ,                                                                                                
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_DPSize,                                                                                                
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Qinv,
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_DPSize);
                    
                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Build_PrivKeyCRT",
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Name,
                                                            TST_MasterTestName);

                }
                
                /* Verify the data using the primes */
                /* (Public Key)                      */
                /*-----------------------------------*/ 
                TST_Status = CRYS_RSA_PRIM_Verify(&TST_PublicKey,
                                                    &TST_PrimeData,
                                                    TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_input_data,
                                                    TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize,
                                                    TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_output_data);    

                TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_PRIM_Verify",
                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Name,
                                                        TST_MasterTestName);

                    
                /*Print the outputted and then the expected result before mem compare */
                ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                        "\nData outputted by engine [output buffer]-- \n",
                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_output_data,
                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize);

                ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                        "\nData Expected by engine [expected buffer - Verify]-- \n",
                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Expected_output_data,
                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize);                                          

                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"\n\n"));
                
                
                /* Compare the Encrypted data to the expected */
                /*--------------------------------------------*/
                TST_Status = (CRYSError_t)DX_VOS_MemCmp(TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_output_data ,
                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Expected_output_data,
                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize);                  

                TST_ATP_CHECK_ERROR_RETURN(TST_Status,"DX_VOS_MemCmp - Verify",
                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Name,
                                                        TST_MasterTestName);

                
                /* Sign the data using the primes */
                /* (Private Key)                     */
                /*-----------------------------------*/     
                TST_Status = CRYS_RSA_PRIM_Sign(&TST_PrivateKey,
                                                &TST_PrimeData,
                                                    TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_output_data,
                                                    TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize,
                                                    TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_output_data);
                                                  
                TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_PRIM_Sign",
                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Name,
                                                        TST_MasterTestName);
 
                                                                
                /*Print the outputted and then the expected result before mem compare */
                ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                        "\nData outputted by engine [output buffer]-- \n",
                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_output_data,
                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize);  

                ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                        "\nData Expected by engine [Input buffer - Sign]-- \n",
                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_input_data,
                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize);                                                                                    
                                            
                TST_Status = (CRYSError_t)DX_VOS_MemCmp(TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_output_data ,
                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_input_data,
                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize);  

               TST_ATP_CHECK_ERROR_RETURN(TST_Status,"DX_VOS_MemCmp - Sign",
                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Name,
                                                        TST_MasterTestName);
                                                                                            
            }/*End of CRT constarint*/
        }/*End of TST_Index loop*/
    }/*End of TST_CRT_Mode loop*/
  return TST_Status; 
   
EXIT_ON_ERROR:
  {
   return DX_FAILURE;
  }
}/*End of CRYS_ATP_TST_RSA_Acceptance_PRIME_SignVerify*/
/*****************************************************/
/*TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************/
/*--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************/
/*TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************/
/*--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************/
/*****************************************************/
/***************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_RSA_Acceptance_PKCS1v15_EncDec         
 * 
 * Inputs:
 *   TST_KeyGeneration_TestStatus - A parameter defining if to run the test with internal key generation 
 *                                TST_INTERNAL_KG - Run the test with internal key generation and external keys 
 *                                TST_EXTERNAL_KEYPAIR  - Run the test with external key only - NO keyGen
 *
 * Outputs:
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for RSA PKCS1v15 encrypt and decrypt
 *  According to PKCS#1 Ver1.5
 *  Including RSA KeyGeneration
 *  This function Encrypts and Decrypts known vectors
 *  Vector/Keys sizes used (in bits) - 512/1024/2048 also CRT private                                                       
 *  This also outputs the results to an output buffer or the same input buffer
 * Algorithm:   
 *  1. Prepare keys 
 *     a. Generate keys 
 *     or
 *     a. Build the Private Key from external exponent and modulus
 *     b. Build the Public Key from External exponent and modulus
 *  2. Encrypt/Decrypt a known vector
 *  3. Compare the outputted result to the expected one                  
 *
 **************************/                                
CRYSError_t CRYS_ATP_TST_RSA_Acceptance_PKCS1v15_EncDec (int TST_KeyGeneration_TestStatus)
{
    /*** TST Variables Definitions ****/
    int         TST_index,TST_CRT_Mode,TST_KeyGenType;
    CRYSError_t TST_Status=0;
    char        TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_RSA_Acceptance_PKCS1v15_EncDec";  
    DxUint16_t  TST_ActualDecDataSize; 

    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"*********************** \n"));
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_RSA_Acceptance_PKCS1v15_EncDec Tests   ******* \n"));       
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"*********************** \n \n"));
    
    /****************/
    /***** Start of test ********/
    /****************/
    /* Setting if to use internal key generation */
    if (TST_KeyGeneration_TestStatus == TST_NO_INTERNAL_KEYGEN)
        TST_KeyGenType = TST_EXTERNAL_KEYPAIR;
    else
        TST_KeyGenType = TST_INTERNAL_KG;

    for ( ; TST_KeyGenType <2 ; TST_KeyGenType++)/*TST_KeyGenType = 0 Internal KeyGen // TST_KeyGenType = 1 External keys*/
    {   
        for (TST_CRT_Mode = 0 ; TST_CRT_Mode < 2 ; TST_CRT_Mode++)/* TST_CRT_Mode = 0 no CRT // TST_CRT_Mode = 1 CRT mode*/
        {   
            for (TST_index =0 ; TST_index < TST_NumberOf_RSA_AcceptanceTests ; TST_index++) /*To run on all test vectors*/
            {
                if (!((TST_CRT_Mode == TST_CRT_MODE) && 
                    (TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize == TST_RSA_512_KEY_SIZE_IN_BYTES)))
                { 
                    TST_ActualDecDataSize = (DxUint16_t)(TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize*8);

                    /*Cleaning the keys to prevent test errors with KG*/ 
                    DX_VOS_MemSet(&TST_PrivateKey,0,sizeof(TST_PrivateKey));
                    DX_VOS_MemSet(&TST_PublicKey,0,sizeof(TST_PublicKey));
                        
                    /* output buffer != input buffer    */
                    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"************************ \n"));
                    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," %s[test 3.%d]\n",TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Name,TST_index));  
                    if (TST_CRT_Mode == TST_NO_CRT)
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," Non CRT mode  \n"));
                    else
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," CRT mode  \n"));
                    if (TST_KeyGenType == TST_INTERNAL_KG)
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," Internal Key Generation  \n"));
                    else
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," External Keys  \n"));                        
                    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"************************ \n"));
                    
                    if (TST_KeyGenType == TST_INTERNAL_KG)
                    {
                        if (TST_CRT_Mode == TST_NO_CRT)
                            {
                                /* Generate The Public and private keys */
                                /**************/
                                TST_Status = CRYS_RSA_KG_GenerateKeyPair(TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_PublicExponent_E,
                                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_PubExponentSize,
                                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize*8,
                                                                            &TST_PrivateKey,
                                                                            &TST_PublicKey,
                                                                            &TST_KeyGenData);
                                                                        
                                TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_KG_GenerateKeyPair",
                                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Name,
                                                                        TST_MasterTestName);
                                                                      
                            }
                            else
                            {
                                /* Generate The Public and private CRT keys */
                                /****************/
                                TST_Status = CRYS_RSA_KG_GenerateKeyPairCRT(TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_PublicExponent_E,
                                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_PubExponentSize,
                                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize*8,
                                                                            &TST_PrivateKey,
                                                                            &TST_PublicKey,
                                                                            &TST_KeyGenData);
                                                                        
                                TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_KG_GenerateKeyPairCRT",
                                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Name,
                                                                        TST_MasterTestName);
                                                             
                            }
                    }
                    else
                    {
                        /* Build the public Key */
                        /*----------------------*/
                        TST_Status = CRYS_RSA_Build_PubKey(&TST_PublicKey,
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_PublicExponent_E,
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_PubExponentSize,
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Modulus_N,
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize);
                                                        
                       TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Build_PubKey",
                                                                TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Name,
                                                                TST_MasterTestName);

                                    
                        /* Build the Private Key */
                        /*-----------------------*/ 
                        if (TST_CRT_Mode == TST_NO_CRT)
                        {
                            TST_Status = CRYS_RSA_Build_PrivKey(&TST_PrivateKey,
                                                                TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_PrivetExponent_D,
                                                                TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize,
                                                                TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_PublicExponent_E,
                                                                TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_PubExponentSize,                                                   
                                                                TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Modulus_N,
                                                                TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize);                                  
                            
                            TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Build_PrivKey",
                                                                    TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Name,
                                                                    TST_MasterTestName);

                        }
                        else
                        {
                            TST_Status = CRYS_RSA_Build_PrivKeyCRT(&TST_PrivateKey,
                                                                    TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_P,
                                                                    TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_DPSize,
                                                                    TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Q,                                                                                                 
                                                                    TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_DPSize,
                                                                    TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_dP,                                                                                                
                                                                    TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_DPSize, 
                                                                    TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_dQ,                                                                                                
                                                                    TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_DPSize,                                                                                                
                                                                    TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Qinv,
                                                                    TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_DPSize);
                            
                            TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Build_PrivKeyCRT",
                                                                    TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Name,
                                                                    TST_MasterTestName);

                        }
                    }/*End of TST_KeyGenType*/
                    
                    /* Encrypt the data using the PKCS1v15 */
                    /* (Public Key)                        */
                    /*-------------------------------------*/   
                    TST_Status = CRYS_RSA_PKCS1v15_Encrypt(&TST_PublicKey,
                                                            &TST_PrimeData,
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_input_data,
                                                            (DxUint16_t)(TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize-TST_LESS_DATA_FOR_PKCS1V15_ENCRYPT),
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_output_data);

                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_PKCS1v15_Encrypt",
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Name,
                                                            TST_MasterTestName);

                
                    /*Print the outputted result */
                    ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                            "\nData outputted by engine [output buffer]-- \n",
                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_output_data,
                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize);              
                            
                    /* Decrypt the data using the PKCS1v15 */
                    /* (Private Key)                       */
                    /*-------------------------------------*/       
                    TST_Status = CRYS_RSA_PKCS1v15_Decrypt(&TST_PrivateKey,
                                                            &TST_PrimeData,                                                             
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_output_data,
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize,
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_output_data,
                                                            &TST_ActualDecDataSize);

                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_PKCS1v15_Decrypt",
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Name,
                                                            TST_MasterTestName);
  
                                                                    
                    /*Print the outputted and then the expected result before mem compare */
                    ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                            "\nData outputted by engine [output buffer]-- \n",
                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_output_data,
                                            TST_ActualDecDataSize);           

                    ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                            "\nData Expected by engine [Input buffer - Decrypt]-- \n",
                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_input_data,
                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize-TST_LESS_DATA_FOR_PKCS1V15_ENCRYPT);                                                                                     
                                                 
                    TST_Status = (CRYSError_t)DX_VOS_MemCmp(TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_output_data ,
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_input_data,
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize-TST_LESS_DATA_FOR_PKCS1V15_ENCRYPT);   

                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"DX_VOS_MemCmp - Decrypt",
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Name,
                                                            TST_MasterTestName);
                                                                                                      
                }/* End of CRT constarint */
            }/*End of TST_Index loop*/
        }/*End of TST_CRT_Mode loop*/
     }/*End of TST_KeyGenType loop*/
  return TST_Status; 
   
EXIT_ON_ERROR:
  {
   return DX_FAILURE;
  }
}/*End of CRYS_ATP_TST_RSA_Acceptance_PKCS1v15_EncDec*/
/*****************************************************/
/*TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************/
/*--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************/
/*TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************/
/*--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************/
/*****************************************************/
/***************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_RSA_Acceptance_PKCS1v15_NonIntegrate_SignVerify        
 * 
 * Inputs:
 *   TST_KeyGeneration_TestStatus - A parameter defining if to run the test with internal key generation 
 *                                TST_INTERNAL_KG - Run the test with internal key generation and external keys 
 *                                TST_EXTERNAL_KEYPAIR  - Run the test with external key only - NO keyGen
 *
 * Outputs:
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for RSA PKCS1v15 Sign & Verify                   
 *  This function Signs & Verifies known vectors using RSA PKCS1v15
 *  Including RSA KeyGeneration
 *  Vectors sizes used (in byte) - 16/32/512/1536                                                       
 *  Key sizes used     (in bits) - 512/1024/2048
 *  This also outputs the results to an output buffer or the same input buffer
 * Algorithm:       
 *  1. Prepare keys 
 *     a. Generate keys 
 *     or
 *     a. Build the Private Key from external exponent and modulus
 *     b. Build the Public Key from External exponent and modulus
 *  2. Call the CRYS_RSA_PKCS1v15_SignInit to init the Sign operation
 *  3. Call the CRYS_RSA_PKCS1v15_SignUpdate - This function is called N times
 *                                        (N = number of blocks in data each block is 16byte)
 *  4. Call the RSA_SignFinish to complete the sign operation
 *  5. Call the CRYS_RSA_PKCS1v15_VerifyInit to init the verification operation
 *  7. Call the CRYS_RSA_PKCS1v15_VerifyUpdate - This function is called N times
 *                                        (N = number of blocks in data each block is 16byte)
 *  7.Call the CRYS_RSA_PKCS1v15_VerifyFinish to complete the verification operation            
 *
 **************************/
CRYSError_t CRYS_ATP_TST_RSA_Acceptance_PKCS1v15_NonIntegrate_SignVerify (int TST_KeyGeneration_TestStatus)
{
    /*** TST Variables Definitions ****/
    int         TST_index,Block_index,TST_CRT_Mode,NumOfBlocks;
    CRYSError_t TST_Status=0;
    int         TST_KeyGenType;
    char        TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_RSA_Acceptance_PKCS1v15_NonIntegrate_SignVerify";   
    int      TST_HashType;
    CRYS_RSA_HASH_OpMode_t    TST_RSAHashType=CRYS_RSA_HASH_NumOfModes;

    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"*********************** \n"));
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_RSA_Acceptance_PKCS1v15_NonIntegrate_SignVerify Tests  ** \n"));      
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"*********************** \n \n"));
    
    /****************/
    /***** Start of test ********/
    /****************/
    /* Setting if to use internal key generation */
    if (TST_KeyGeneration_TestStatus == TST_NO_INTERNAL_KEYGEN)
        TST_KeyGenType = TST_EXTERNAL_KEYPAIR;
    else
        TST_KeyGenType = TST_INTERNAL_KG;

    for ( ; TST_KeyGenType <2 ; TST_KeyGenType++)/*TST_KeyGenType = 0 Internal KeyGen // TST_KeyGenType = 1 External keys*/
    {   
        for (TST_HashType = 0 ; TST_HashType < CRYS_HASH_NumOfModes ; TST_HashType++) 
        {       
            switch(TST_HashType)
            {
            case CRYS_HASH_SHA1_mode:
                TST_RSAHashType = CRYS_RSA_HASH_SHA1_mode;
                break;
            case CRYS_HASH_SHA224_mode:
                TST_RSAHashType = CRYS_RSA_HASH_SHA224_mode;
                break;
            case CRYS_HASH_SHA256_mode:
                TST_RSAHashType = CRYS_RSA_HASH_SHA256_mode;
                break;
            case CRYS_HASH_SHA384_mode:
                #ifndef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
                  TST_RSAHashType = CRYS_RSA_HASH_SHA384_mode;
                  break;
                #else
                  continue;
                #endif
            case CRYS_HASH_SHA512_mode:
                #ifndef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
                  TST_RSAHashType = CRYS_RSA_HASH_SHA512_mode;
                  break;
                #else
                  continue;
                #endif
            default:
                break;
            }
            for (TST_CRT_Mode = 0 ; TST_CRT_Mode < 2 ; TST_CRT_Mode++)/* TST_CRT_Mode = 0 no CRT // TST_CRT_Mode = 1 CRT mode*/
            {   
                for (TST_index =0 ; TST_index < TST_NumberOf_RSA_AcceptanceSignVerifyTests ; TST_index++) /*To run on all test vectors*/
                {
                    /* CRT constraint CRT with 512 key should not work */
                    if (((TST_CRT_Mode == TST_CRT_MODE) && 
                        (TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_KeySize == TST_RSA_512_KEY_SIZE_IN_BYTES)))
                        continue;
                    /* SHA384 & SHA512 are not used with RSA 512bit keys*/
                    if ((TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_KeySize == TST_RSA_512_KEY_SIZE_IN_BYTES)&&
                        ((TST_RSAHashType == CRYS_RSA_HASH_SHA384_mode)||(TST_RSAHashType == CRYS_RSA_HASH_SHA512_mode)))
                        continue;
                    
                    /*Cleaning the keys to prevent test errors with KG*/ 
                    DX_VOS_MemSet(&TST_PrivateKey,0,sizeof(TST_PrivateKey));
                    DX_VOS_MemSet(&TST_PublicKey,0,sizeof(TST_PublicKey));
                            
                    /* output buffer != input buffer    */
                    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"************************ \n"));
                    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," %s[test 4.%d]\n",TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,TST_index)); 
                    CRYS_ATP_TST_PrintHASHType((CRYS_HASH_OperationMode_t)TST_HashType);              
                    if (TST_CRT_Mode == TST_NO_CRT)
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," Non CRT mode  \n"));
                    else
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," CRT mode  \n"));
                    if (TST_KeyGenType == TST_INTERNAL_KG)
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," Internal Key Generation  \n"));
                    else
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," External Keys  \n"));                        
                    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"************************ \n"));

                    
                    NumOfBlocks = TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_KeySize/TST_MAX_SINGLE_RSA_SIGN_BLOCK_SIZE_IN_BYTS;
                    
                    if (TST_KeyGenType == TST_INTERNAL_KG)
                    {
                        if (TST_CRT_Mode == TST_NO_CRT)
                            {
                                /* Generate The Public and private keys */
                                /**************/
                                TST_Status = CRYS_RSA_KG_GenerateKeyPair(TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_PublicExponent_E,
                                                                            TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_PubExponentSize,
                                                                            TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_KeySize*8,
                                                                            &TST_PrivateKey,
                                                                            &TST_PublicKey,
                                                                            &TST_KeyGenData);
                                                                        
                                TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_KG_GenerateKeyPair",
                                                                        TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,
                                                                        TST_MasterTestName);
                                                                          
                            }
                            else
                            {
                                /* Generate The Public and private CRT keys */
                                /****************/
                                TST_Status = CRYS_RSA_KG_GenerateKeyPairCRT(TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_PublicExponent_E,
                                                                            TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_PubExponentSize,
                                                                            TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_KeySize*8,
                                                                            &TST_PrivateKey,
                                                                            &TST_PublicKey,
                                                                            &TST_KeyGenData);
                                                                        
                                TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_KG_GenerateKeyPairCRT",
                                                                        TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,
                                                                        TST_MasterTestName);
                                                                    
                            }
                    }
                    else
                    {
                        /* Build the public Key */
                        /*----------------------*/
                        TST_Status = CRYS_RSA_Build_PubKey(&TST_PublicKey,
                                                            TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_PublicExponent_E,
                                                            TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_PubExponentSize,
                                                            TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Modulus_N,
                                                            TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_KeySize);
                                                         
                                                    
                        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Build_PubKey",
                                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,
                                                                TST_MasterTestName);
         
                                    
                        /* Build the Private Key */
                        /*-----------------------*/ 
                        if (TST_CRT_Mode == TST_NO_CRT)
                        {
                            TST_Status = CRYS_RSA_Build_PrivKey(&TST_PrivateKey,
                                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_PrivetExponent_D,
                                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_KeySize,
                                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_PublicExponent_E,
                                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_PubExponentSize,                                                  
                                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Modulus_N,
                                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_KeySize);                                 
                            
                            TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Build_PrivKey",
                                                                    TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,
                                                                    TST_MasterTestName);

                        }
                        else
                        {
                            TST_Status = CRYS_RSA_Build_PrivKeyCRT(&TST_PrivateKey,
                                                                    TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_P,
                                                                    TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_DPSize,
                                                                    TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Q,                                                                                                
                                                                    TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_DPSize,
                                                                    TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_dP,                                                                                                   
                                                                    TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_DPSize,    
                                                                    TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_dQ,                                                                                                   
                                                                    TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_DPSize,                                                                                                   
                                                                    TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Qinv,
                                                                    TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_DPSize);
                                                                                           
                            TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Build_PrivKeyCRT",
                                                                    TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,
                                                                    TST_MasterTestName);

                        }
                    }/* End of TST_KeyGenType*/
                    
                    /*Init the Sign operation */
                    /*------------------------*/
                    TST_Status = CRYS_RSA_PKCS1v15_SignInit(&TST_ContextPrivate,
                                                               &TST_PrivateKey,
                                                               TST_RSAHashType);
 
                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_PKCS1v15_SignInit",
                                                            TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,
                                                            TST_MasterTestName);
                                       
                    
                    /* Sign the data using the private Key */
                    /* Calling the update function         */
                    /*-------------------------------------*/
                    NumOfBlocks = TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_input_dataSize/TST_MAX_SINGLE_RSA_SIGN_BLOCK_SIZE_IN_BYTS;
                    
                    /* Debug massages */
                    ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"number of blocks = %1d \n",NumOfBlocks));
                    ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"Input Data Size [byte] = %1d \n" ,TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_input_dataSize));
                    ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"SingleBlockSize [byte] = %1d \n",TST_MAX_SINGLE_RSA_SIGN_BLOCK_SIZE_IN_BYTS));
                    
                    
                    for (Block_index = 0 ; Block_index < NumOfBlocks ; Block_index++)/*RSA PSS_SignUpdate block loop*/
                    { 
                        TST_Status = CRYS_RSA_PKCS1v15_SignUpdate(&TST_ContextPrivate,
                                                                    TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_input_data+(TST_MAX_SINGLE_RSA_SIGN_BLOCK_SIZE_IN_BYTS*Block_index),
                                                                    TST_MAX_SINGLE_RSA_SIGN_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t));                                        
                                                
                        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_PKCS1v15_SignUpdate",
                                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,
                                                                TST_MasterTestName);
                                      
                    }                       
                    
                    /* Finish the Sign operation */
                    /*---------------------------*/
                    TST_Status = CRYS_RSA_PKCS1v15_SignFinish(&TST_ContextPrivate,
                                                                 TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_output_signiture,
                                   &TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_KeySize);
                            
                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_PKCS1v15_SignFinish",
                                                              TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,
                                                              TST_MasterTestName);
 

                    /*Print the outputted signature*/
                    ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                            "The outputted signature - \n",
                                            TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_output_signiture,
                                            TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_KeySize);

                    /*Init the Verify operation */
                    /*--------------------------*/
                    TST_Status = CRYS_RSA_PKCS1v15_VerifyInit(&TST_ContextPublic,
                                                                 &TST_PublicKey,
                                                                 TST_RSAHashType);
 
                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_PKCS1v15_VerifyInit",
                                                            TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,
                                                            TST_MasterTestName);
                                       
                    
                    /* Verify the data using the private Key */
                    /* Calling the update function           */
                    /*---------------------------------------*/
                    for (Block_index = 0 ; Block_index < NumOfBlocks ; Block_index++)/*RSA PSS_SignUpdate block loop*/
                    { 
                        CRYS_RSA_PKCS1v15_VerifyUpdate(&TST_ContextPublic,                                     
                                                       TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_input_data+(TST_MAX_SINGLE_RSA_SIGN_BLOCK_SIZE_IN_BYTS*Block_index),
                                                       TST_MAX_SINGLE_RSA_SIGN_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t));
                                                
                        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_PKCS1v15_VerifyUpdate",
                                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,
                                                                TST_MasterTestName);
                                     
                    }                           
                    /* Finish the Verify operation */
                    /*-----------------------------*/   
                    TST_Status = CRYS_RSA_PKCS1v15_VerifyFinish(&TST_ContextPublic,
                                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_output_signiture);
                        
                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_PKCS1v15_VerifyFinish",
                                                            TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,
                                                            TST_MasterTestName);

                }/*End of Index loop*/
            }/*End of TST_CRT_Mode loop*/
         }/*End of TST_HashType loop*/
      }/*End of TST_KeyGenType loop*/
    return TST_Status; 

EXIT_ON_ERROR:
  {
   return DX_FAILURE;
  }
}/*End of CRYS_ATP_TST_RSA_Acceptance_PKCS1v15_NonIntegrate_SignVerify*/
/*****************************************************/
/*TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************/
/*--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************/
/*TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************/
/*--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************/
/*****************************************************/
/***************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_RSA_Acceptance_PKCS1v15_Integrated_SignVerify      
 * 
 * Inputs:
 *   TST_KeyGeneration_TestStatus - A parameter defining if to run the test with internal key generation 
 *                                TST_INTERNAL_KG - Run the test with internal key generation and external keys 
 *                                TST_EXTERNAL_KEYPAIR  - Run the test with external key only - NO keyGen
 *
 * Outputs:
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for RSA PKCS1v15 Sign & Verify                   
 *  This function Signs & Verifies known vectors using RSA PKCS1v15
 *  Vectors sizes used (in byte) - 16/32/512                                                        
 *  Key sizes used     (in bits) - 512/1024/2048
 *  This also outputs the results to an output buffer or the same input buffer
 * Algorithm:       
 *  1. Prepare keys 
 *     a. Generate keys 
 *     or
 *     a. Build the Private Key from external exponent and modulus
 *     b. Build the Public Key from External exponent and modulus
 *  3. Call the CRYS_RSA_PKCS1v15_Sign to preform the Sign operation
 *  4. Call the CRYS_RSA_PKCS1v15_Verify to complete the verification operation
 *  5. Compare the outputted result to the expected one                  
 *
 **************************/
CRYSError_t CRYS_ATP_TST_RSA_Acceptance_PKCS1v15_Integrated_SignVerify (int TST_KeyGeneration_TestStatus)
{
    /*** TST Variables Definitions ****/
    int         TST_index,TST_CRT_Mode;
    CRYSError_t TST_Status=0;
    int         TST_KeyGenType;
    char        TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_RSA_Acceptance_PKCS1v15_Integrated_SignVerify"; 
    int       TST_HashType;
    CRYS_RSA_HASH_OpMode_t TST_RSAHashType=CRYS_RSA_HASH_NumOfModes;

    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"*********************** \n"));
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_RSA_Acceptance_PKCS1v15_Integrated_SignVerify Tests    ** \n"));      
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"*********************** \n \n"));
    
    /****************/
    /***** Start of test ********/
    /****************/
    /* Setting if to use internal key generation */
    if (TST_KeyGeneration_TestStatus == TST_NO_INTERNAL_KEYGEN)
        TST_KeyGenType = TST_EXTERNAL_KEYPAIR;
    else
        TST_KeyGenType = TST_INTERNAL_KG;

    for ( ; TST_KeyGenType <2 ; TST_KeyGenType++)/*TST_KeyGenType = 0 Internal KeyGen // TST_KeyGenType = 1 External keys*/
    {   
        for (TST_HashType = 0 ; TST_HashType < CRYS_HASH_NumOfModes ; TST_HashType++)
        {   
            switch(TST_HashType)
            {
                case CRYS_HASH_SHA1_mode:
                    TST_RSAHashType = CRYS_RSA_HASH_SHA1_mode;
                    break;
                case CRYS_HASH_SHA224_mode:
                    TST_RSAHashType = CRYS_RSA_HASH_SHA224_mode;
                    break;
                case CRYS_HASH_SHA256_mode:
                    TST_RSAHashType = CRYS_RSA_HASH_SHA256_mode;
                    break;
                case CRYS_HASH_SHA384_mode:
                    #ifndef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
                    TST_RSAHashType = CRYS_RSA_HASH_SHA384_mode;
                    break;
                    #else
                    continue;
                    #endif
                case CRYS_HASH_SHA512_mode:
                    #ifndef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
                    TST_RSAHashType = CRYS_RSA_HASH_SHA512_mode;
                    break;
                    #else
                    continue;
                    #endif
                default:
                    break;
            }
            for (TST_CRT_Mode = 0 ; TST_CRT_Mode < 2 ; TST_CRT_Mode++)/* TST_CRT_Mode = 0 no CRT // TST_CRT_Mode = 1 CRT mode*/
            {   
                for (TST_index =0 ; TST_index < TST_NumberOf_RSA_AcceptanceSignVerifyTests ; TST_index++) /*To run on all test vectors*/
                {
                    /* CRT constraint CRT with 512 key should not work */
                    if (((TST_CRT_Mode == TST_CRT_MODE) && 
                        (TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_KeySize == TST_RSA_512_KEY_SIZE_IN_BYTES)))
                    continue;
                    /* SHA384 & SHA512 are not used with RSA 512bit keys*/
                    if ((TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_KeySize == TST_RSA_512_KEY_SIZE_IN_BYTES)&&
                        ((TST_RSAHashType == CRYS_RSA_HASH_SHA384_mode)||(TST_RSAHashType == CRYS_RSA_HASH_SHA512_mode)))
                    continue;
                    
                    /*Cleaning the keys to prevent test errors with KG*/ 
                    DX_VOS_MemSet(&TST_PrivateKey,0,sizeof(TST_PrivateKey));
                    DX_VOS_MemSet(&TST_PublicKey,0,sizeof(TST_PublicKey));
                    
                    /* output buffer != input buffer    */
                    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"************************ \n"));
                    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," %s[test 5.%d]\n",TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,TST_index)); 
                    CRYS_ATP_TST_PrintHASHType((CRYS_HASH_OperationMode_t)TST_HashType);                
                    if (TST_CRT_Mode == TST_NO_CRT)
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," Non CRT mode  \n"));
                    else
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," CRT mode  \n"));
                    if (TST_KeyGenType == TST_INTERNAL_KG)
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," Internal Key Generation  \n"));
                    else
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," External Keys  \n"));                                
                    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"************************ \n"));
            
                    if (TST_KeyGenType == TST_INTERNAL_KG)
                    {
                        if (TST_CRT_Mode == TST_NO_CRT)
                            {
                                /* Generate The Public and private keys */
                                /**************/
                                TST_Status = CRYS_RSA_KG_GenerateKeyPair(TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_PublicExponent_E,
                                                                            TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_PubExponentSize,
                                                                            TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_KeySize*8,
                                                                            &TST_PrivateKey,
                                                                            &TST_PublicKey,
                                                                            &TST_KeyGenData);
                                                                        
                                TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_KG_GenerateKeyPair",
                                                                        TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,
                                                                        TST_MasterTestName);
                                                                      
                            }
                            else
                            {
                                /* Generate The Public and private CRT keys */
                                /****************/
                                TST_Status = CRYS_RSA_KG_GenerateKeyPairCRT(TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_PublicExponent_E,
                                                                            TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_PubExponentSize,
                                                                            TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_KeySize*8,
                                                                            &TST_PrivateKey,
                                                                            &TST_PublicKey,
                                                                            &TST_KeyGenData);
                                                                        
                                TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_KG_GenerateKeyPairCRT",
                                                                        TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,
                                                                        TST_MasterTestName);
                                                              
                            }
                    }
                    else
                    {                    
                        /* Build the public Key */
                        /*----------------------*/
                        TST_Status = CRYS_RSA_Build_PubKey(&TST_PublicKey,
                                                            TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_PublicExponent_E,
                                                            TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_PubExponentSize,
                                                            TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Modulus_N,
                                                            TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_KeySize);
                                                         
                                                    
                        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Build_PubKey",
                                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,
                                                                TST_MasterTestName);
 
                                    
                        /* Build the Private Key */
                        /*-----------------------*/ 
                        if (TST_CRT_Mode == TST_NO_CRT)
                        {
                            TST_Status = CRYS_RSA_Build_PrivKey(&TST_PrivateKey,
                                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_PrivetExponent_D,
                                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_KeySize,
                                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_PublicExponent_E,
                                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_PubExponentSize,                                                  
                                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Modulus_N,
                                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_KeySize);                                 
                            
                            TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Build_PrivKey",
                                                                    TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,
                                                                    TST_MasterTestName);

                        }
                        else
                        {
                            TST_Status = CRYS_RSA_Build_PrivKeyCRT(&TST_PrivateKey,
                                                                    TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_P,
                                                                    TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_DPSize,
                                                                    TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Q,                                                                                                
                                                                    TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_DPSize,
                                                                    TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_dP,                                                                                                   
                                                                    TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_DPSize,    
                                                                    TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_dQ,                                                                                                   
                                                                    TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_DPSize,                                                                                                   
                                                                    TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Qinv,
                                                                    TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_DPSize);
                                                                                           
                            TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Build_PrivKeyCRT",
                                                                    TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,
                                                                    TST_MasterTestName);

                        }
                    }/*End of TST_KeyGenType*/
                    
                    /* Debug massages */
                    ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"Input Data Size [byte] = %1d \n" ,TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_input_dataSize));


                    /*Preform the Sign operation */
                    /*---------------------------*/
                    TST_Status = CRYS_RSA_PKCS1v15_Sign(&TST_ContextPrivate,
                                                           &TST_PrivateKey,
                                                           TST_RSAHashType,
                                                           TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_input_data,
                                                           TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_input_dataSize,
                                                           TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_output_signiture,
                                 &TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_KeySize);                                            
   
                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_PKCS1v15_Sign",
                                                              TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,
                                                              TST_MasterTestName);
                                                
                    
                    /*Print the outputted signature*/
                    ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                            "The outputted signature - \n",
                                            TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_output_signiture,
                                            TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_KeySize);

                    /*Preform the Verify operation */
                    /*-----------------------------*/
                    TST_Status = CRYS_RSA_PKCS1v15_Verify(&TST_ContextPublic,
                                                             &TST_PublicKey,
                                                             TST_RSAHashType,
                                                             TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_input_data,
                                                             TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_input_dataSize,
                                                             TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_output_signiture);                                                

                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_PKCS1v15_Verify - Using SHA1",
                                                            TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,
                                                            TST_MasterTestName);
                                                 
                }/*End of TST_Index loop*/
            }/*End of TST_CRT_Mode loop*/
         }/* End of TST_HashType loop*/
        }/* End of TST_KeyGenType loop*/
     return TST_Status; 

EXIT_ON_ERROR:
  {
   return DX_FAILURE;
  }
}/*End of CRYS_ATP_TST_RSA_Acceptance_PKCS1v15_Integrated_SignVerify*/
/*****************************************************/
/*TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************/
/*--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************/
/*TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************/
/*--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************/
/*****************************************************/
/***************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_RSA_Acceptance_PKCS1v15_ExternalHash_SignVerify        
 * 
 * Inputs:
 *   TST_KeyGeneration_TestStatus - A parameter defining if to run the test with internal key generation 
 *                                TST_INTERNAL_KG - Run the test with internal key generation and external keys 
 *                                TST_EXTERNAL_KEYPAIR  - Run the test with external key only - NO keyGen
 *
 * Outputs:
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for RSA PKCS1v15 Sign & Verify
 *  When the Hash for the Sign is preformed externally                   
 *  This function Signs & Verifies hash vectors using RSA PKCS1v15
 *  Vectors sizes used (in byte) - SHA1 - 20, MD5 - 16                                                      
 *  Key sizes used     (in bits) - 512/1024/2048
 * Algorithm:       
 *  1. Prepare keys 
 *     a. Generate keys 
 *     or
 *     a. Build the Private Key from external exponent and modulus
 *     b. Build the Public Key from External exponent and modulus
 *  2. Call the CRYS_RSA_PKCS1v15_SHA1_Sign or CRYS_RSA_PKCS1v15_MD5_Sign to preform the Sign operation
 *  3. Call the CRYS_RSA_PKCS1v15_SHA1_Verify or CRYS_RSA_PKCS1v15_MD5_verify to complete the verification operation
 *  4. Compare the outputted result to the expected one                  
 *
 **************************/
CRYSError_t CRYS_ATP_TST_RSA_Acceptance_PKCS1v15_ExternalHash_SignVerify (int TST_KeyGeneration_TestStatus)
{
    /*** TST Variables Definitions ****/
    int         TST_index,TST_CRT_Mode;
    CRYSError_t TST_Status=0;
    int         TST_HashType,TST_KeyGenType;
    char        TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_RSA_Acceptance_PKCS1v15_ExternalHash_SignVerify";       
    
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"*********************** \n"));
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_RSA_Acceptance_PKCS1v15_ExternalHash_SignVerify Tests  ** \n"));      
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"*********************** \n \n"));
    
    /****************/
    /***** Start of test ********/
    /****************/
    /* Setting if to use internal key generation */
    if (TST_KeyGeneration_TestStatus == TST_NO_INTERNAL_KEYGEN)
        TST_KeyGenType = TST_EXTERNAL_KEYPAIR;
    else
        TST_KeyGenType = TST_INTERNAL_KG;

    for ( ; TST_KeyGenType <2 ; TST_KeyGenType++)/*TST_KeyGenType = 0 Internal KeyGen // TST_KeyGenType = 1 External keys*/
    {       
        for (TST_HashType = 0 ; TST_HashType < CRYS_HASH_NumOfModes ; TST_HashType++) 
        {
            for (TST_CRT_Mode = 0 ; TST_CRT_Mode < 2 ; TST_CRT_Mode++)/* TST_CRT_Mode = 0 no CRT // TST_CRT_Mode = 1 CRT mode*/
            {   
                for (TST_index =0 ; TST_index < TST_NumberOf_RSA_AcceptanceExtHashSignVerifyTests ; TST_index++) /*To run on all test vectors*/
                {
                    /* CRT constraint CRT with 512 key should not work */
                    if ((TST_CRT_Mode == TST_CRT_MODE) && 
                        (TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_KeySize == TST_RSA_512_KEY_SIZE_IN_BYTES))
                        continue;
                    /* SHA384 & SHA512 are not used with RSA 512bit keys*/
                    if ((TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_KeySize == TST_RSA_512_KEY_SIZE_IN_BYTES)&&
                        ((TST_HashType == CRYS_HASH_SHA384_mode)||(TST_HashType == CRYS_HASH_SHA512_mode)))
                        continue;
                    
                    /*Cleaning the keys to prevent test errors with KG*/ 
                    DX_VOS_MemSet(&TST_PrivateKey,0,sizeof(TST_PrivateKey));
                    DX_VOS_MemSet(&TST_PublicKey,0,sizeof(TST_PublicKey));
                    
                    /* output buffer != input buffer    */
                    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"************************ \n"));
                    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," %s[test 6.%d]\n",TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_Name,TST_index));  
                    CRYS_ATP_TST_PrintHASHType((CRYS_HASH_OperationMode_t)TST_HashType);
                    if (TST_CRT_Mode == TST_NO_CRT)
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," Non CRT mode  \n"));
                    else
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," CRT mode  \n"));
                    if (TST_KeyGenType == TST_INTERNAL_KG)
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," Internal Key Generation  \n"));
                    else
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," External Keys  \n"));                            
                    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"************************ \n"));
                
                    if (TST_KeyGenType == TST_INTERNAL_KG)
                    {
                        if (TST_CRT_Mode == TST_NO_CRT)
                            {
                                /* Generate The Public and private keys */
                                /**************/
                                TST_Status = CRYS_RSA_KG_GenerateKeyPair(TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_PublicExponent_E,
                                                                            TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_PubExponentSize,
                                                                            TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_KeySize*8,
                                                                            &TST_PrivateKey,
                                                                            &TST_PublicKey,
                                                                            &TST_KeyGenData);
                                                                        
                                TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_KG_GenerateKeyPair",
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_Name,
                                                                        TST_MasterTestName);
                                                                    
                            }
                            else
                            {
                                /* Generate The Public and private CRT keys */
                                /****************/
                                TST_Status = CRYS_RSA_KG_GenerateKeyPairCRT(TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_PublicExponent_E,
                                                                            TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_PubExponentSize,
                                                                            TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_KeySize*8,
                                                                            &TST_PrivateKey,
                                                                            &TST_PublicKey,
                                                                            &TST_KeyGenData);
                                                                        
                                TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_KG_GenerateKeyPairCRT",
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_Name,
                                                                        TST_MasterTestName);
                                                             
                            }
                    }
                    else
                    {
                        /* Build the public Key */
                        /*----------------------*/
                        TST_Status = CRYS_RSA_Build_PubKey(&TST_PublicKey,
                                                            TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_PublicExponent_E,
                                                            TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_PubExponentSize,
                                                            TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_Modulus_N,
                                                            TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_KeySize);
                                                         
                                                    
                        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Build_PubKey",
                                                                TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_Name,
                                                                TST_MasterTestName);
  
                                    
                        /* Build the Private Key */
                        /*-----------------------*/ 
                        if (TST_CRT_Mode == TST_NO_CRT)
                        {
                            TST_Status = CRYS_RSA_Build_PrivKey(&TST_PrivateKey,
                                                                TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_PrivetExponent_D,
                                                                TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_KeySize,
                                                                TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_PublicExponent_E,
                                                                TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_PubExponentSize,
                                                                TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_Modulus_N,
                                                                TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_KeySize);                                  
                            
                            TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Build_PrivKey",TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_Name,TST_MasterTestName);

                        }
                        else
                        {
                            TST_Status = CRYS_RSA_Build_PrivKeyCRT(&TST_PrivateKey,
                                                                    TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_P,
                                                                    TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_DPSize,
                                                                    TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_Q,                                                                                                 
                                                                    TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_DPSize,
                                                                    TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_dP,                                                                                                
                                                                    TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_DPSize, 
                                                                    TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_dQ,                                                                                                
                                                                    TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_DPSize,                                                                                                
                                                                    TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_Qinv,
                                                                    TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_DPSize);
                                                                                           
                            TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Build_PrivKeyCRT",
                                                                    TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_Name,
                                                                    TST_MasterTestName);

                        }
                    }/* End of TST_KeyGenType*/
                    
                    /*Preform the Sign operation */
                    /*---------------------------*/
                    switch(TST_HashType)
                    {
                        case CRYS_HASH_SHA1_mode:
                            TST_Status = CRYS_RSA_PKCS1v15_SHA1_Sign(&TST_ContextPrivate,
                                                                        &TST_PrivateKey,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_input_hash_SHA1,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_output_signiture,
                                          &TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_KeySize); 
                            break;
                        case CRYS_HASH_SHA224_mode:
                            TST_Status = CRYS_RSA_PKCS1v15_SHA224_Sign(&TST_ContextPrivate,
                                                                        &TST_PrivateKey,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_input_hash_SHA224,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_output_signiture,
                                          &TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_KeySize); 
                            break;
                        case CRYS_HASH_SHA256_mode:
                            TST_Status = CRYS_RSA_PKCS1v15_SHA256_Sign(&TST_ContextPrivate,
                                                                        &TST_PrivateKey,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_input_hash_SHA256,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_output_signiture,
                                          &TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_KeySize); 
                            break;
                        case CRYS_HASH_SHA384_mode:
                            #ifndef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
                            TST_Status = CRYS_RSA_PKCS1v15_SHA384_Sign(&TST_ContextPrivate,
                                                                        &TST_PrivateKey,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_input_hash_SHA384,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_output_signiture,
                                          &TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_KeySize); 
                            break;
              #else
              continue;
              #endif
                        case CRYS_HASH_SHA512_mode:
                            #ifndef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
                            TST_Status = CRYS_RSA_PKCS1v15_SHA512_Sign(&TST_ContextPrivate,
                                                                        &TST_PrivateKey,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_input_hash_SHA512,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_output_signiture,
                                          &TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_KeySize); 
                            break;
                            #else
                            continue;
                            #endif
                        case CRYS_HASH_MD5_mode:    
                            TST_Status = CRYS_RSA_PKCS1v15_MD5_Sign(&TST_ContextPrivate,
                                                                        &TST_PrivateKey,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_input_hash_MD5,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_output_signiture,
                                          &TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_KeySize); 
                            break;
                        default:
                            break;
                    }
                                                            
                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_PKCS1v15_SHA1_Sign",
                                                            TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_Name,
                                                            TST_MasterTestName);

                                                                                                                                                                                                                                        
                    /*Print the outputted signature*/
                    ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                            "The outputted signature - \n",
                                            TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_output_signiture,
                                            TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_KeySize);

                    /*Preform the Verify operation */
                    /*-----------------------------*/
                    switch(TST_HashType)
                    {
                        case CRYS_HASH_SHA1_mode:
                            TST_Status = CRYS_RSA_PKCS1v15_SHA1_Verify(&TST_ContextPublic,
                                                                        &TST_PublicKey,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_input_hash_SHA1,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_output_signiture);   
                            break;
                        case CRYS_HASH_SHA224_mode:
                            TST_Status = CRYS_RSA_PKCS1v15_SHA224_Verify(&TST_ContextPublic,
                                                                        &TST_PublicKey,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_input_hash_SHA224,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_output_signiture);   
                            break;
                        case CRYS_HASH_SHA256_mode:
                            TST_Status = CRYS_RSA_PKCS1v15_SHA256_Verify(&TST_ContextPublic,
                                                                        &TST_PublicKey,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_input_hash_SHA256,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_output_signiture);   
                            break;
                        case CRYS_HASH_SHA384_mode:
                            #ifndef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
                            TST_Status = CRYS_RSA_PKCS1v15_SHA384_Verify(&TST_ContextPublic,
                                                                        &TST_PublicKey,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_input_hash_SHA384,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_output_signiture);   
                            break;
                            #else
                            continue;
                            #endif
                        case CRYS_HASH_SHA512_mode:
                            #ifndef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
                            TST_Status = CRYS_RSA_PKCS1v15_SHA512_Verify(&TST_ContextPublic,
                                                                        &TST_PublicKey,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_input_hash_SHA512,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_output_signiture);   
                            break;
                            #else
                            continue;
                            #endif
                        case CRYS_HASH_MD5_mode:  
                            TST_Status = CRYS_RSA_PKCS1v15_MD5_Verify(&TST_ContextPublic,
                                                                        &TST_PublicKey,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_input_hash_MD5,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_output_signiture);   
                            break;
                        default:
                            break;
                    }     

                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_PKCS1v15_Verify",
                                                            TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_Name,
                                                            TST_MasterTestName);
                                  
                                   

                }/*End of TST_Index loop*/
            }/*End of TST_CRT_Mode loop*/
         }/*End of TST_HashType loop*/
      }/* End of TST_KeyGenType loop*/  
  return TST_Status; 
   
EXIT_ON_ERROR:
  {
   return DX_FAILURE;
  }
}/*End of CRYS_ATP_TST_RSA_Acceptance_PKCS1v15_ExternalHash_SignVerify*/
/*****************************************************/
/*TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************/
/*--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************/
/*TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************/
/*--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************/
/*****************************************************/
/***************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_RSA_Acceptance_OAEP_EncDec         
 * 
 * Inputs:
 *   TST_KeyGeneration_TestStatus - A parameter defining if to run the test with internal key generation 
 *                                TST_INTERNAL_KG - Run the test with internal key generation and external keys 
 *                                TST_EXTERNAL_KEYPAIR  - Run the test with external key only - NO keyGen
 *
 * Outputs:
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for RSA OAEP encrypt and decrypt
 *  According to PKCS#1 Ver2.1
 *  This function Encrypts and Decrypts known vectors
 *  Vector/Keys sizes used (in bits) - 512/1024/2048                                                        
 *  This also outputs the results to an output buffer or the same input buffer
 * Algorithm:       
 *  1. Prepare keys 
 *     a. Generate keys 
 *     or
 *     a. Build the Private Key from external exponent and modulus
 *     b. Build the Public Key from External exponent and modulus       
 *  2. Encrypt/Decrypt a known vector
 *  3. Compare the outputted result to the expected one                  
 *
 **************************/                                
CRYSError_t CRYS_ATP_TST_RSA_Acceptance_OAEP_EncDec (int TST_KeyGeneration_TestStatus)
{
    /*** TST Variables Definitions ****/
    int           TST_index,TST_CRT_Mode;
    CRYSError_t   TST_Status=0;
    int           TST_KeyGenType;
    char          TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_RSA_Acceptance_OAEP_EncDec";  
    DxUint16_t    TST_ActualDecDataSize;
    DxUint16_t    TST_LessDataForOAEP=0;

    int        TST_HashType;
    CRYS_RSA_HASH_OpMode_t    TST_RSAHashType=CRYS_RSA_HASH_NumOfModes;
    
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"*********************** \n"));
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_RSA_Acceptance_OAEP_EncDec Tests   ******* \n"));       
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"*********************** \n \n"));
    
    /****************/
    /***** Start of test ********/
    /****************/
    /* Setting if to use internal key generation */
    if (TST_KeyGeneration_TestStatus == TST_NO_INTERNAL_KEYGEN)
        TST_KeyGenType = TST_EXTERNAL_KEYPAIR;
    else
        TST_KeyGenType = TST_INTERNAL_KG;

    for ( ; TST_KeyGenType <2 ; TST_KeyGenType++)/*TST_KeyGenType = 0 Internal KeyGen // TST_KeyGenType = 1 External keys*/
    {      
        for (TST_HashType = 0 ; TST_HashType < CRYS_HASH_NumOfModes ; TST_HashType++) 
        {
            switch(TST_HashType)
            {
                case CRYS_HASH_SHA1_mode:
                    TST_RSAHashType = CRYS_RSA_HASH_SHA1_mode;
                    TST_LessDataForOAEP = (DxUint16_t)TST_LESS_DATA_FOR_OAEP_ENCRYPT_SHA1;
                    break;
                case CRYS_HASH_SHA224_mode:
                    TST_RSAHashType = CRYS_RSA_HASH_SHA224_mode;
                    TST_LessDataForOAEP = (DxUint16_t)TST_LESS_DATA_FOR_OAEP_ENCRYPT_SHA224;
                    break;
                case CRYS_HASH_SHA256_mode:
                    TST_RSAHashType = CRYS_RSA_HASH_SHA256_mode;
                    TST_LessDataForOAEP = (DxUint16_t)TST_LESS_DATA_FOR_OAEP_ENCRYPT_SHA256;
                    break;
                case CRYS_HASH_SHA384_mode:
                    #ifndef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
                    TST_RSAHashType = CRYS_RSA_HASH_SHA384_mode;
                    TST_LessDataForOAEP = (DxUint16_t)TST_LESS_DATA_FOR_OAEP_ENCRYPT_SHA384;
                    break;
                    #else
                    continue;
                    #endif
                case CRYS_HASH_SHA512_mode:
                    #ifndef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
                    TST_RSAHashType = CRYS_RSA_HASH_SHA512_mode;
                    TST_LessDataForOAEP = (DxUint16_t)TST_LESS_DATA_FOR_OAEP_ENCRYPT_SHA512;
                    break;
                    #else
                    continue;
                    #endif
                default:
                    break;
            }
            if (TST_HashType != CRYS_HASH_MD5_mode) /* MD5 is not a part of the OAEP scheme*/
            {
                for (TST_CRT_Mode = 0 ; TST_CRT_Mode < 2 ; TST_CRT_Mode++)/* TST_CRT_Mode = 0 no CRT // TST_CRT_Mode = 1 CRT mode*/
                {   
                    for (TST_index =0 ; TST_index < TST_NumberOf_RSA_AcceptanceTests ; TST_index++) /*To run on all test vectors*/
                    {
                        /* CRT constraint CRT with 512 key should not work */
                        if ((TST_CRT_Mode == TST_CRT_MODE) && 
                            (TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize == TST_RSA_512_KEY_SIZE_IN_BYTES))
                            continue;
                        /* SHA256 & SHA384 & SHA512 are not used with RSA 512bit keys in OAEP*/
                        if ((TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize == TST_RSA_512_KEY_SIZE_IN_BYTES)&&
                            ((TST_RSAHashType == CRYS_RSA_HASH_SHA256_mode)||(TST_RSAHashType == CRYS_RSA_HASH_SHA384_mode)||(TST_RSAHashType == CRYS_RSA_HASH_SHA512_mode)))
                            continue;
						/* SHA512 is not used with RSA 1024bit keys in OAEP*/
						if ((TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize == TST_RSA_1024_KEY_SIZE_IN_BYTES)&&
							(TST_RSAHashType == CRYS_RSA_HASH_SHA512_mode))
							continue;
                        
                        TST_ActualDecDataSize = (DxUint16_t)(TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize*8);
                        /*Cleaning the keys to prevent test errors with KG*/ 
                        DX_VOS_MemSet(&TST_PrivateKey,0,sizeof(TST_PrivateKey));
                        DX_VOS_MemSet(&TST_PublicKey,0,sizeof(TST_PublicKey));         

                        /* output buffer != input buffer    */
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"************************ \n"));
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," %s[test 7.%d]\n",TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Name,TST_index));  
                        if (TST_CRT_Mode == TST_NO_CRT)
                            ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," Non CRT mode  \n"));
                        else
                            ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," CRT mode  \n"));
                        if (TST_KeyGenType == TST_INTERNAL_KG)
                            ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," Internal Key Generation  \n"));
                        else
                            ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," External Keys  \n")); 
                        CRYS_ATP_TST_PrintHASHType((CRYS_HASH_OperationMode_t)TST_HashType);
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"************************ \n"));
                        
                        if (TST_KeyGenType == TST_INTERNAL_KG)
                        {
                            if (TST_CRT_Mode == TST_NO_CRT)
                                {
                                    /* Generate The Public and private keys */
                                    /**************/
                                    TST_Status = CRYS_RSA_KG_GenerateKeyPair(TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_PublicExponent_E,
                                                                                TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_PubExponentSize,
                                                                                TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize*8,
                                                                                &TST_PrivateKey,
                                                                                &TST_PublicKey,
                                                                                &TST_KeyGenData);
                                                                            
                                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_KG_GenerateKeyPair",
                                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Name,
                                                                            TST_MasterTestName);
                                                                      
                                }
                                else
                                {
                                    /* Generate The Public and private CRT keys */
                                    /****************/
                                    TST_Status = CRYS_RSA_KG_GenerateKeyPairCRT(TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_PublicExponent_E,
                                                                                TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_PubExponentSize,
                                                                                TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize*8,
                                                                                &TST_PrivateKey,
                                                                                &TST_PublicKey,
                                                                                &TST_KeyGenData);
                                                                            
                                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_KG_GenerateKeyPairCRT",
                                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Name,
                                                                            TST_MasterTestName);
                                                             
                                }
                        }
                        else
                        {
                            /* Build the public Key */
                            /*----------------------*/
                            TST_Status = CRYS_RSA_Build_PubKey(&TST_PublicKey,
                                                                TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_PublicExponent_E,
                                                                TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_PubExponentSize,
                                                                TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Modulus_N,
                                                                TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize);
                                                            
                            TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Build_PubKey",
                                                                    TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Name,
                                                                    TST_MasterTestName);
  
                                        
                            /* Build the Private Key */
                            /*-----------------------*/ 
                            if (TST_CRT_Mode == TST_NO_CRT)
                            {
                                TST_Status = CRYS_RSA_Build_PrivKey(&TST_PrivateKey,
                                                                    TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_PrivetExponent_D,
                                                                    TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize,
                                                                    TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_PublicExponent_E,
                                                                    TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_PubExponentSize,
                                                                    TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Modulus_N,
                                                                    TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize);                                  
                                
                                TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Build_PrivKey",
                                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Name,
                                                                        TST_MasterTestName);

                            }
                            else
                            {
                                TST_Status = CRYS_RSA_Build_PrivKeyCRT(&TST_PrivateKey,
                                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_P,
                                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_DPSize,
                                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Q,                                                                                                 
                                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_DPSize,
                                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_dP,                                                                                                
                                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_DPSize, 
                                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_dQ,                                                                                                
                                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_DPSize,                                                                                                
                                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Qinv,
                                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_DPSize);
                                
                                TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Build_PrivKeyCRT",
                                                                        TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Name,
                                                                        TST_MasterTestName);

                            }
                        }/* End of TST_KeyGenType*/
                        
                        /* Encrypt the data using the OAEP */
                        /* (Public Key)                    */
                        /*---------------------------------*/   
                        TST_Status = CRYS_RSA_OAEP_Encrypt(&TST_PublicKey,
                                                              &TST_PrimeData,
                                                              TST_RSAHashType,
                                                              DX_NULL,
                                                              TST_ZERO_DATA,
                                                              CRYS_PKCS1_MGF1,
                                                              TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_input_data,
                                                              (DxUint16_t)(TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize-TST_LessDataForOAEP),
                                                              TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_output_data);                                                                                                      

                        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_OAEP_Encrypt",
                                                                TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Name,
                                                                TST_MasterTestName);

                        
                        /*Print the outputted result */
                        ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                                "\nData outputted by engine [output buffer]-- \n",
                                                TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_output_data,
                                                TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize);              
                                
                        /* Decrypt the data using the OAEP */
                        /* (Private Key)                   */
                        /*---------------------------------*/       
                        TST_Status = CRYS_RSA_OAEP_Decrypt(&TST_PrivateKey,
                                                            &TST_PrimeData,
                                                            TST_RSAHashType,               
                                                            DX_NULL,
                                                            TST_ZERO_DATA,
                                                            CRYS_PKCS1_MGF1,                                            
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_output_data,
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize,
                                                            TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_output_data,
                                                            &TST_ActualDecDataSize);

                        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_OAEP_Decrypt",
                                                                TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Name,
                                                                TST_MasterTestName);
 
                                                                        
                        /*Print the outputted and then the expected result before mem compare */
                        ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                                "\nData outputted by engine [output buffer]-- \n",
                                                TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_output_data,
                                                TST_ActualDecDataSize);           

                        ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                                "\nData Expected by engine [Input buffer - Decrypt]-- \n",
                                                TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_input_data,
                                                TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize-TST_LessDataForOAEP);                                                                                     
                                                     
                        TST_Status = (CRYSError_t)DX_VOS_MemCmp(TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_output_data ,
                                                                TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_input_data,
                                                                TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_KeySize-TST_LessDataForOAEP);   

                        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"DX_VOS_MemCmp - Decrypt",
                                                                TST_RSA_Acceptance_Primitive_TestData[TST_index].TST_Name,
                                                                TST_MasterTestName);

                    }/*End of TST_Index loop*/
                }/*End of TST_CRT_Mode loop*/
            }/*End of MD5 if*/
        }/*End of Hash Type loop*/
     }/*End of TST_KeyGenType loop*/    
  return TST_Status; 
   
EXIT_ON_ERROR:
  {
   return DX_FAILURE;
  }
}/*End of CRYS_ATP_TST_RSA_Acceptance_OAEP_EncDec*/
/*****************************************************/
/*TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************/
/*--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************/
/*TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************/
/*--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************/
/*****************************************************/
/***************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_RSA_Acceptance_PSS_NonIntegrate_SignVerify         
 * 
 * Inputs:
 *   TST_KeyGeneration_TestStatus - A parameter defining if to run the test with internal key generation 
 *                                TST_INTERNAL_KG - Run the test with internal key generation and external keys 
 *                                TST_EXTERNAL_KEYPAIR  - Run the test with external key only - NO keyGen
 *
 * Outputs:
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for RSA PSS PKCS#1v2.1 sign & Verify                     
 *  This function  Signs & Verifies known vectors using RSA PSS PKCS#1v2.1
 *  Vectors sizes used (in byte) - 16/32/512                                                        
 *  Key sizes used     (in bits) - 512/1024/2048
 *  This also outputs the results to an output buffer or the same input buffer
 * Algorithm:       
 *  1. Prepare keys 
 *     a. Generate keys 
 *     or
 *     a. Build the Private Key from external exponent and modulus
 *     b. Build the Public Key from External exponent and modulus   
 *  2. Call the CRYS_RSA_PSS_SignInit to init the Sign operation
 *  3. Call the CRYS_RSA_PSS_SignUpdate - This function is called N times
 *                                        (N = number of blocks in data each block is 16byte)
 *  4. Call the RSA_SignFinish to complete the sign operation
 *  5. Call the CRYS_RSA_PSS_VerifyInit to init the verification operation
 *  6. Call the CRYS_RSA_PSS_VerifyUpdate - This function is called N times
 *                                        (N = number of blocks in data each block is 16byte)
 *  7.Call the CRYS_RSA_PSS_VerifyFinish to complete the verification operation             
 *
 **************************/                                
CRYSError_t CRYS_ATP_TST_RSA_Acceptance_PSS_NonIntegrate_SignVerify (int TST_KeyGeneration_TestStatus)
{
    /*** TST Variables Definitions ****/
    int         TST_index,Block_index,TST_CRT_Mode,NumOfBlocks;
    CRYSError_t TST_Status=0;
    int         TST_KeyGenType;
    char        TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_RSA_Acceptance_PSS_NonIntegrate_SignVerify";    
    int       TST_HashType;
    CRYS_RSA_HASH_OpMode_t    TST_RSAHashType=CRYS_RSA_HASH_NumOfModes;
    
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"*********************** \n"));
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_RSA_Acceptance_PSS_NonIntegrate_SignVerify Tests  *** \n"));      
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"*********************** \n \n"));
    
    /****************/
    /***** Start of test ********/
    /****************/
    /* Setting if to use internal key generation */
    if (TST_KeyGeneration_TestStatus == TST_NO_INTERNAL_KEYGEN)
        TST_KeyGenType = TST_EXTERNAL_KEYPAIR;
    else
        TST_KeyGenType = TST_INTERNAL_KG;

    for ( ; TST_KeyGenType <2 ; TST_KeyGenType++)/*TST_KeyGenType = 0 Internal KeyGen // TST_KeyGenType = 1 External keys*/
    {   
        for (TST_HashType = 0 ; TST_HashType < CRYS_HASH_NumOfModes ; TST_HashType++) 
        {
            switch(TST_HashType)
            {
                case CRYS_HASH_SHA1_mode:
                    TST_RSAHashType = CRYS_RSA_HASH_SHA1_mode;
                    break;
                case CRYS_HASH_SHA224_mode:
                    TST_RSAHashType = CRYS_RSA_HASH_SHA224_mode;
                    break;
                case CRYS_HASH_SHA256_mode:
                    TST_RSAHashType = CRYS_RSA_HASH_SHA256_mode;
                    break;
                case CRYS_HASH_SHA384_mode:
                    #ifndef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
                    TST_RSAHashType = CRYS_RSA_HASH_SHA384_mode;
                    break;
                    #else
                    continue;
                    #endif
                case CRYS_HASH_SHA512_mode:
                    #ifndef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
                    TST_RSAHashType = CRYS_RSA_HASH_SHA512_mode;
                    break;
                    #else
                    continue;
                    #endif
                default:
                    break;
            }
            if (TST_HashType != CRYS_HASH_MD5_mode) /* MD5 is not a part of the OAEP scheme*/
            {
                for (TST_CRT_Mode = 0 ; TST_CRT_Mode < 2 ; TST_CRT_Mode++)/* TST_CRT_Mode = 0 no CRT // TST_CRT_Mode = 1 CRT mode*/
                {   
                    for (TST_index =0 ; TST_index < TST_NumberOf_RSA_AcceptanceSignVerifyTests ; TST_index++) /*To run on all test vectors*/
                    {
                        /* CRT constraint CRT with 512 key should not work */
                        if ((TST_CRT_Mode == TST_CRT_MODE) && 
                            (TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_KeySize == TST_RSA_512_KEY_SIZE_IN_BYTES))
                            continue;
                        /* SHA384 & SHA512 are not used with RSA 512bit keys*/
                        if ((TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_KeySize == TST_RSA_512_KEY_SIZE_IN_BYTES)&&
                            ((TST_RSAHashType == CRYS_RSA_HASH_SHA384_mode)||(TST_RSAHashType == CRYS_RSA_HASH_SHA512_mode)))
                            continue;
                        
                        /*Cleaning the keys to prevent test errors with KG*/ 
                        DX_VOS_MemSet(&TST_PrivateKey,0,sizeof(TST_PrivateKey));
                        DX_VOS_MemSet(&TST_PublicKey,0,sizeof(TST_PublicKey));         
                        
                        /* output buffer != input buffer    */
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"************************ \n"));
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," %s[test 8.%d]\n",TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,TST_index)); 
                        if (TST_CRT_Mode == TST_NO_CRT)
                            ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," Non CRT mode  \n"));
                        else
                            ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," CRT mode  \n"));
                        if (TST_KeyGenType == TST_INTERNAL_KG)
                            ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," Internal Key Generation  \n"));
                        else
                            ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," External Keys  \n"));  
                        CRYS_ATP_TST_PrintHASHType((CRYS_HASH_OperationMode_t)TST_HashType);
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"************************ \n"));

                        
                        NumOfBlocks = TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_KeySize/TST_MAX_SINGLE_RSA_SIGN_BLOCK_SIZE_IN_BYTS;          
                        
                        if (TST_KeyGenType == TST_INTERNAL_KG)
                        {
                            if (TST_CRT_Mode == TST_NO_CRT)
                                {
                                    /* Generate The Public and private keys */
                                    /**************/
                                    TST_Status = CRYS_RSA_KG_GenerateKeyPair(TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_PublicExponent_E,
                                                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_PubExponentSize,
                                                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_KeySize*8,
                                                                                &TST_PrivateKey,
                                                                                &TST_PublicKey,
                                                                                &TST_KeyGenData);
                                                                            
                                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_KG_GenerateKeyPair",
                                                                            TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,
                                                                            TST_MasterTestName);
                                                                     
                                }
                                else
                                {
                                    /* Generate The Public and private CRT keys */
                                    /****************/
                                    TST_Status = CRYS_RSA_KG_GenerateKeyPairCRT(TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_PublicExponent_E,
                                                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_PubExponentSize,
                                                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_KeySize*8,
                                                                                &TST_PrivateKey,
                                                                                &TST_PublicKey,
                                                                                &TST_KeyGenData);
                                                                            
                                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_KG_GenerateKeyPairCRT",
                                                                            TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,
                                                                            TST_MasterTestName);
                                                           
                                }
                        }
                        else
                        {
                            /* Build the public Key */
                            /*----------------------*/
                            TST_Status = CRYS_RSA_Build_PubKey(&TST_PublicKey,
                                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_PublicExponent_E,
                                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_PubExponentSize,
                                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Modulus_N,
                                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_KeySize);
                                                             
                                                        
                            TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Build_PubKey",
                                                                    TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,
                                                                    TST_MasterTestName);
 
                                        
                            /* Build the Private Key */
                            /*-----------------------*/ 
                            if (TST_CRT_Mode == TST_NO_CRT)
                            {
                                TST_Status = CRYS_RSA_Build_PrivKey(&TST_PrivateKey,
                                                                       TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_PrivetExponent_D,
                                                                       TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_KeySize,
                                                                       TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_PublicExponent_E,
                                                                       TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_PubExponentSize,                                                  
                                                                       TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Modulus_N,
                                                                       TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_KeySize);                                 
                                
                                TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Build_PrivKey",
                                                                        TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,
                                                                        TST_MasterTestName);

                            }
                            else
                            {
                                TST_Status = CRYS_RSA_Build_PrivKeyCRT(&TST_PrivateKey,
                                                                          TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_P,
                                                                          TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_DPSize,
                                                                          TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Q,                                                                                                
                                                                          TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_DPSize,
                                                                          TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_dP,                                                                                                   
                                                                          TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_DPSize,    
                                                                          TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_dQ,                                                                                                   
                                                                          TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_DPSize,                                                                                                   
                                                                          TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Qinv,
                                                                          TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_DPSize);
                                                                                               
                                TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Build_PrivKeyCRT",
                                                                        TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,
                                                                        TST_MasterTestName);

                            
                            }
                        }/* End of TST_KeyGenType*/
                        
                        /*Init the Sign operation */
                        /*------------------------*/
                        TST_Status = CRYS_RSA_PSS_SignInit(&TST_ContextPrivate,
                                                              &TST_PrivateKey,
                                                              TST_RSAHashType,
                                                              CRYS_PKCS1_MGF1,
                                                              TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_SaltLength);
                                                          
                        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_PSS_SignInit",
                                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,
                                                                TST_MasterTestName);
                                            
                        
                        /* Sign the data using the private Key */
                        /* Calling the update function         */
                        /*-------------------------------------*/
                        NumOfBlocks = TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_input_dataSize/TST_MAX_SINGLE_RSA_SIGN_BLOCK_SIZE_IN_BYTS;
                        
                        /* Debug massages */
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"number of blocks = %1d \n",NumOfBlocks));
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"Input Data Size [byte] = %1d \n" ,TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_input_dataSize));
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"SingleBlockSize [byte] = %1d \n",TST_MAX_SINGLE_HASH_BLOCK_SIZE_IN_BYTS));
                        
                        
                        for (Block_index = 0 ; Block_index < NumOfBlocks ; Block_index++)/*RSA PSS_SignUpdate block loop*/
                        { 
                            TST_Status = CRYS_RSA_PSS_SignUpdate(&TST_ContextPrivate,
                                                                    TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_input_data+(TST_MAX_SINGLE_RSA_SIGN_BLOCK_SIZE_IN_BYTS*Block_index),
                                                                    TST_MAX_SINGLE_RSA_SIGN_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t));                                        
                                                    
                            TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_PSS_SignUpdate",
                                                                    TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,
                                                                    TST_MasterTestName);
                                      
                        }                       

                                
                        /* Finish the Sign operation */
                        /*---------------------------*/ 
                        TST_Status = CRYS_RSA_PSS_SignFinish(&TST_ContextPrivate,
                                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_output_signiture,
                                                                &TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_KeySize);
                            
                       TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_PSS_SignFinish",
                                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,
                                                                TST_MasterTestName);
 
                        
                        /*Print the outputted signature*/
                        ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                                "The outputted signature - \n",
                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_output_signiture,
                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_KeySize);

                        /*Init the Verify operation */
                        /*--------------------------*/
                        TST_Status = CRYS_RSA_PSS_VerifyInit(&TST_ContextPublic,
                                                                &TST_PublicKey,
                                                                TST_RSAHashType,
                                                                CRYS_PKCS1_MGF1,
                                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_SaltLength);
                                                          
                        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_PSS_VerifyInit",
                                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,
                                                                TST_MasterTestName);
                                           
                        
                        /* Verify the data using the private Key */
                        /* Calling the update function           */
                        /*---------------------------------------*/
                        for (Block_index = 0 ; Block_index < NumOfBlocks ; Block_index++)/*RSA PSS_SignUpdate block loop*/
                        { 
                            TST_Status = CRYS_RSA_PSS_VerifyUpdate(&TST_ContextPublic,
                                                                      TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_input_data+(TST_MAX_SINGLE_RSA_SIGN_BLOCK_SIZE_IN_BYTS*Block_index),
                                                                      TST_MAX_SINGLE_RSA_SIGN_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t));                                        
                                                    
                            TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_PSS_VerifyUpdate",
                                                                      TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,
                                                                      TST_MasterTestName);
                                    
                        }                           
                        /* Finish the Verify operation */
                        /*-----------------------------*/   
                        TST_Status = CRYS_RSA_PSS_VerifyFinish(&TST_ContextPublic,
                                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_output_signiture);
                            
                        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_PSS_VerifyFinish",TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,TST_MasterTestName);
 
                    }/*End of TST_Index loop*/
                }/*End of TST_CRT_Mode loop*/
            }/*If condition for MD5*/
        }/*Loop Of HASH type*/
     }/* End of TST_KeyGenType loop*/
  return TST_Status; 
   
EXIT_ON_ERROR:
  {
   return DX_FAILURE;
  }
}/*End of CRYS_ATP_TST_RSA_Acceptance_PSS_NonIntegrate_SignVerify*/
/*****************************************************/
/*TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************/
/*--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************/
/*TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************/
/*--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************/
/*****************************************************/
/***************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_RSA_Acceptance_PSS_Integrated_SignVerify       
 * 
 * Inputs:
 *   TST_KeyGeneration_TestStatus - A parameter defining if to run the test with internal key generation 
 *                                TST_INTERNAL_KG - Run the test with internal key generation and external keys 
 *                                TST_EXTERNAL_KEYPAIR  - Run the test with external key only - NO keyGen
 *
 * Outputs:
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for RSA PSS PKCS#1v2.1 sign & Verify                     
 *  This function  Signs & Verifies known vectors using RSA PSS PKCS#1v2.1
 *  Vectors sizes used (in byte) - 16/32/512 /1536                                                       
 *  Key sizes used     (in bits) - 512/1024/2048
 *  This also outputs the results to an output buffer or the same input buffer
 * Algorithm:       
 *  1. Prepare keys 
 *     a. Generate keys 
 *     or
 *     a. Build the Private Key from external exponent and modulus
 *     b. Build the Public Key from External exponent and modulus   
 *  2. Call the CRYS_RSA_PSS_Sign to preform the Sign operation
 *  3. Call the RSA_PSS_Verify to preform the verification operation                    
 *
 **************************/                                
CRYSError_t CRYS_ATP_TST_RSA_Acceptance_PSS_Integrated_SignVerify (int TST_KeyGeneration_TestStatus)
{
    /*** TST Variables Definitions ****/
    int         TST_index,TST_CRT_Mode;
    CRYSError_t TST_Status=0;
    int         TST_KeyGenType;
    char        TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_RSA_Acceptance_PSS_Integrated_SignVerify";  
    int        TST_HashType;
    CRYS_RSA_HASH_OpMode_t    TST_RSAHashType=CRYS_RSA_HASH_NumOfModes;
    
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"*********************** \n"));
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_RSA_Acceptance_PSS_Integrated_SignVerify Tests         ** \n"));      
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"*********************** \n \n"));
    
    /****************/
    /***** Start of test ********/
    /****************/
    /* Setting if to use internal key generation */
    if (TST_KeyGeneration_TestStatus == TST_NO_INTERNAL_KEYGEN)
        TST_KeyGenType = TST_EXTERNAL_KEYPAIR;
    else
        TST_KeyGenType = TST_INTERNAL_KG;

    for ( ; TST_KeyGenType <2 ; TST_KeyGenType++)/*TST_KeyGenType = 0 Internal KeyGen // TST_KeyGenType = 1 External keys*/
    { 
        for (TST_HashType = 0 ; TST_HashType < CRYS_HASH_NumOfModes ; TST_HashType++) 
        {
            switch(TST_HashType)
            {
                case CRYS_HASH_SHA1_mode:
                    TST_RSAHashType = CRYS_RSA_HASH_SHA1_mode;
                    break;
                case CRYS_HASH_SHA224_mode:
                    TST_RSAHashType = CRYS_RSA_HASH_SHA224_mode;
                    break;
                case CRYS_HASH_SHA256_mode:
                    TST_RSAHashType = CRYS_RSA_HASH_SHA256_mode;
                    break;
                case CRYS_HASH_SHA384_mode:
                    #ifndef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
                    TST_RSAHashType = CRYS_RSA_HASH_SHA384_mode;
                    break;
                    #else
                    continue;
                    #endif
                case CRYS_HASH_SHA512_mode:
                    #ifndef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
                    TST_RSAHashType = CRYS_RSA_HASH_SHA512_mode;
                    break;
                    #else
                    continue;
                    #endif
                default:
                    break;
            }
            if (TST_HashType != CRYS_HASH_MD5_mode) /* MD5 is not a part of the OAEP scheme*/
            {
                for (TST_CRT_Mode = 0 ; TST_CRT_Mode < 2 ; TST_CRT_Mode++)/* TST_CRT_Mode = 0 no CRT // TST_CRT_Mode = 1 CRT mode*/
                {   
                    for (TST_index =0 ; TST_index < TST_NumberOf_RSA_AcceptanceSignVerifyTests ; TST_index++) /*To run on all test vectors*/
                    {
                        /* CRT constraint CRT with 512 key shoud not work */
                        if ((TST_CRT_Mode == TST_CRT_MODE) && 
                            (TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_KeySize == TST_RSA_512_KEY_SIZE_IN_BYTES))
                            continue;
                        /* SHA384 & SHA512 are not used with RSA 512bit keys*/
                        if ((TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_KeySize == TST_RSA_512_KEY_SIZE_IN_BYTES)&&
                            ((TST_HashType == CRYS_HASH_SHA384_mode)||(TST_HashType == CRYS_HASH_SHA512_mode)))
                            continue;
                        
                        /*Cleaning the keys to prevent test errors with KG*/ 
                        DX_VOS_MemSet(&TST_PrivateKey,0,sizeof(TST_PrivateKey));
                        DX_VOS_MemSet(&TST_PublicKey,0,sizeof(TST_PublicKey));         
                                                
                        /* output buffer != input buffer    */
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"************************ \n"));
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," %s[test 9.%d]\n",TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,TST_index)); 
                        if (TST_CRT_Mode == TST_NO_CRT)
                            ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," Non CRT mode  \n"));
                        else
                            ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," CRT mode  \n"));
                        if (TST_KeyGenType == TST_INTERNAL_KG)
                            ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," Internal Key Generation  \n"));
                        else
                            ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," External Keys  \n"));    
                        CRYS_ATP_TST_PrintHASHType((CRYS_HASH_OperationMode_t)TST_HashType);
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"************************ \n"));     

                        if (TST_KeyGenType == TST_INTERNAL_KG)
                        {
                            if (TST_CRT_Mode == TST_NO_CRT)
                                {
                                    /* Generate The Public and private keys */
                                    /**************/
                                    TST_Status = CRYS_RSA_KG_GenerateKeyPair(TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_PublicExponent_E,
                                                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_PubExponentSize,
                                                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_KeySize*8,
                                                                                &TST_PrivateKey,
                                                                                &TST_PublicKey,
                                                                                &TST_KeyGenData);
                                                                            
                                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_KG_GenerateKeyPair",
                                                                            TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,
                                                                            TST_MasterTestName);
                                                                      
                                }
                                else
                                {
                                    /* Generate The Public and private CRT keys */
                                    /****************/
                                    TST_Status = CRYS_RSA_KG_GenerateKeyPairCRT(TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_PublicExponent_E,
                                                                                    TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_PubExponentSize,
                                                                                    TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_KeySize*8,
                                                                                    &TST_PrivateKey,
                                                                                    &TST_PublicKey,
                                                                                    &TST_KeyGenData);
                                                                            
                                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_KG_GenerateKeyPairCRT",
                                                                            TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,
                                                                            TST_MasterTestName);
                                                             
                                }
                        }
                        else
                        {               
                            /* Build the public Key */
                            /*----------------------*/
                            TST_Status = CRYS_RSA_Build_PubKey(&TST_PublicKey,
                                                                  TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_PublicExponent_E,
                                                                  TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_PubExponentSize,
                                                                  TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Modulus_N,
                                                                  TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_KeySize);
                                                               
                                                        
                            TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Build_PubKey",
                                                                    TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,
                                                                    TST_MasterTestName);
 
                                        
                            /* Build the Private Key */
                            /*-----------------------*/ 
                            if (TST_CRT_Mode == TST_NO_CRT)
                            {
                                TST_Status = CRYS_RSA_Build_PrivKey(&TST_PrivateKey,
                                                                        TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_PrivetExponent_D,
                                                                        TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_KeySize,
                                                                        TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_PublicExponent_E,
                                                                        TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_PubExponentSize,                                                  
                                                                        TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Modulus_N,
                                                                        TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_KeySize);                                 
                                    
                                TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Build_PrivKey",
                                                                        TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,
                                                                        TST_MasterTestName);

                            }
                            else
                            {
                                TST_Status = CRYS_RSA_Build_PrivKeyCRT(&TST_PrivateKey,
                                                                          TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_P,
                                                                          TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_DPSize,
                                                                          TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Q,                                                                                                
                                                                          TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_DPSize,
                                                                          TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_dP,                                                                                                   
                                                                          TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_DPSize,    
                                                                          TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_dQ,                                                                                                   
                                                                          TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_DPSize,                                                                                                   
                                                                          TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Qinv,
                                                                          TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_DPSize);
                                                                                                 
                                TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Build_PrivKeyCRT",
                                                                        TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,
                                                                        TST_MasterTestName);

                            
                            }
                        }/*End of TST_KeyGenType*/
                        
                        /*Preform the Sign operation */
                        /*---------------------------*/
                        TST_Status = CRYS_RSA_PSS_Sign(&TST_ContextPrivate,
                                                          &TST_PrivateKey,
                                                          TST_RSAHashType,
                                                          CRYS_PKCS1_MGF1,
                                                          TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_SaltLength,
                                                          TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_input_data,
                                                          TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_input_dataSize,
                                                          TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_output_signiture,
                                                          &TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_KeySize);
                      
                        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_PSS_Sign",
                                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,
                                                                TST_MasterTestName);
                                           
                                    
                        /*Print the outputted signature*/
                        ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                                "The outputted signature - \n",
                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_output_signiture,
                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_KeySize);

                        /*Preform the Verify operation */
                        /*-----------------------------*/
                        TST_Status = CRYS_RSA_PSS_Verify(&TST_ContextPublic,
                                                            &TST_PublicKey,
                                                            TST_RSAHashType,
                                                            CRYS_PKCS1_MGF1,
                                                            TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_SaltLength,
                                                            TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_input_data,
                                                            TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_input_dataSize,
                                                            TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_output_signiture);                                        
                                                          
                        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_PSS_Verify",
                                                                TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,
                                                                TST_MasterTestName);
 
                    }/*End of TST_Index loop*/
                }/*End of TST_CRT_Mode loop*/
            }/*End of if MD5*/
        }/*End of HASH type loop*/
     }/*End of TST_KeyGenType Loop*/
  return TST_Status; 
   
EXIT_ON_ERROR:
  {
   return DX_FAILURE;
  }
}/*End of CRYS_ATP_TST_RSA_Acceptance_PSS_NonIntegrate_SignVerify*/
/*****************************************************/
/*TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************/
/*--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************/
/*TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************/
/*--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************/
/*****************************************************/
/***************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_RSA_Acceptance_PSS_ExternalHash_SignVerify         
 * 
 * Inputs:
 *   TST_KeyGeneration_TestStatus - A parameter defining if to run the test with internal key generation 
 *                                TST_INTERNAL_KG - Run the test with internal key generation and external keys 
 *                                TST_EXTERNAL_KEYPAIR  - Run the test with external key only - NO keyGen
 *
 * Outputs:
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for RSA PKCS1v2.1 PSS Sign & Verify
 *  When the Hash for the Sign is preformed externally                   
 *  This function Signs & Verifies hash vectors using RSA PKCS1v2.1 PSS                                                   
 *  Key sizes used     (in bits) - 512/1024/2048
 * Algorithm:       
 *  1. Prepare keys 
 *     a. Generate keys 
 *     or
 *     a. Build the Private Key from external exponent and modulus
 *     b. Build the Public Key from External exponent and modulus 
 *  2. Call the CRYS_RSA_PSS_SHA1_Sign or CRYS_RSA_PSS_MD5_Sign to preform the Sign operation
 *  3. Call the CRYS_RSA_PSS_SHA1_Verify or CRYS_RSA_PSS_MD5_verify to complete the verification operation
 *  4. Compare the outputted result to the expected one                  
 *
 **************************/
CRYSError_t CRYS_ATP_TST_RSA_Acceptance_PSS_ExternalHash_SignVerify (int TST_KeyGeneration_TestStatus)
{
    /*** TST Variables Definitions ****/
    int         TST_index,TST_CRT_Mode;
    CRYSError_t TST_Status=0;
    int         TST_KeyGenType;
    char        TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_RSA_Acceptance_PSS_ExternalHash_SignVerify";    
    int      TST_HashType;
    
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"*********************** \n"));
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_RSA_Acceptance_PSS_ExternalHash_SignVerify Tests       ** \n"));      
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"*********************** \n \n"));
    
    /****************/
    /***** Start of test ********/
    /****************/
    /* Setting if to use internal key generation */
    if (TST_KeyGeneration_TestStatus == TST_NO_INTERNAL_KEYGEN)
        TST_KeyGenType = TST_EXTERNAL_KEYPAIR;
    else
        TST_KeyGenType = TST_INTERNAL_KG;

    for ( ; TST_KeyGenType <2 ; TST_KeyGenType++)/*TST_KeyGenType = 0 Internal KeyGen // TST_KeyGenType = 1 External keys*/
    {   
        for (TST_HashType = 0 ; TST_HashType < CRYS_HASH_NumOfModes ; TST_HashType++) 
        {
            if (TST_HashType != CRYS_HASH_MD5_mode) /* MD5 is not a part of the OAEP scheme*/
            {
                for (TST_CRT_Mode = 0 ; TST_CRT_Mode < 2 ; TST_CRT_Mode++)/* TST_CRT_Mode = 0 no CRT // TST_CRT_Mode = 1 CRT mode*/
                {   
                    for (TST_index =0 ; TST_index < TST_NumberOf_RSA_AcceptanceExtHashSignVerifyTests ; TST_index++) /*To run on all test vectors*/
                    {
                        /* CRT constraint CRT with 512 key should not work */
                        if ((TST_CRT_Mode == TST_CRT_MODE) && 
                            (TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_KeySize == TST_RSA_512_KEY_SIZE_IN_BYTES))
                            continue;
                        /* SHA384 & SHA512 are not used with RSA 512bit keys*/
                        if ((TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_KeySize == TST_RSA_512_KEY_SIZE_IN_BYTES)&&
                            ((TST_HashType == CRYS_HASH_SHA384_mode)||(TST_HashType == CRYS_HASH_SHA512_mode)))
                            continue;
                        
                        /*Cleaning the keys to prevent test errors with KG*/ 
                        DX_VOS_MemSet(&TST_PrivateKey,0,sizeof(TST_PrivateKey));
                        DX_VOS_MemSet(&TST_PublicKey,0,sizeof(TST_PublicKey));         
                        
                        /* output buffer != input buffer    */
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"************************ \n"));
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," %s[test A.%d]\n",TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_Name,TST_index));  
                        if (TST_CRT_Mode == TST_NO_CRT)
                            ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," Non CRT mode  \n"));
                        else
                            ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," CRT mode  \n"));
                        if (TST_KeyGenType == TST_INTERNAL_KG)
                            ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," Internal Key Generation  \n"));
                        else
                            ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," External Keys  \n"));  
                        CRYS_ATP_TST_PrintHASHType((CRYS_HASH_OperationMode_t)TST_HashType);
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"************************ \n")); 
                        

                        if (TST_KeyGenType == TST_INTERNAL_KG)
                        {
                            if (TST_CRT_Mode == TST_NO_CRT)
                                {
                                    /* Generate The Public and private keys */
                                    /**************/
                                    TST_Status = CRYS_RSA_KG_GenerateKeyPair(TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_PublicExponent_E,
                                                                                TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_PubExponentSize,
                                                                                TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_KeySize*8,
                                                                                &TST_PrivateKey,
                                                                                &TST_PublicKey,
                                                                                &TST_KeyGenData);
                                                                            
                                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_KG_GenerateKeyPair",
                                                               TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_Name,
                                                               TST_MasterTestName);

                                }
                                else
                                {
                                    /* Generate The Public and private CRT keys */
                                    /****************/
                                    TST_Status = CRYS_RSA_KG_GenerateKeyPairCRT(TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_PublicExponent_E,
                                                                                  TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_PubExponentSize,
                                                                                  TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_KeySize*8,
                                                                                  &TST_PrivateKey,
                                                                                  &TST_PublicKey,
                                                                                  &TST_KeyGenData);
                                                                            
                                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_KG_GenerateKeyPairCRT",
                                                                TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_Name,
                                                                TST_MasterTestName);
                                }
                        }
                        else
                        {                   
                            /* Build the public Key */
                            /*----------------------*/
                            TST_Status = CRYS_RSA_Build_PubKey(&TST_PublicKey,
                                                                  TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_PublicExponent_E,
                                                                  TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_PubExponentSize,
                                                                  TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_Modulus_N,
                                                                  TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_KeySize);
                                                               
                                                        
                            TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Build_PubKey",
                                                                    TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_Name,
                                                                    TST_MasterTestName);
 
                                        
                            /* Build the Private Key */
                            /*-----------------------*/ 
                            if (TST_CRT_Mode == TST_NO_CRT)
                            {
                                TST_Status = CRYS_RSA_Build_PrivKey(&TST_PrivateKey,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_PrivetExponent_D,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_KeySize,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_PublicExponent_E,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_PubExponentSize,                                               
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_Modulus_N,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_KeySize);                                  
                                    
                                TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Build_PrivKey",
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_Name,
                                                                        TST_MasterTestName);

                            }
                            else
                            {
                                TST_Status = CRYS_RSA_Build_PrivKeyCRT(&TST_PrivateKey,
                                                                          TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_P,
                                                                          TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_DPSize,
                                                                          TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_Q,                                                                                                 
                                                                          TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_DPSize,
                                                                          TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_dP,                                                                                                
                                                                          TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_DPSize, 
                                                                          TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_dQ,                                                                                                
                                                                          TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_DPSize,                                                                                                
                                                                          TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_Qinv,
                                                                          TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_DPSize);
                                                                                               
                                TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Build_PrivKeyCRT",
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_Name,
                                                                        TST_MasterTestName);

                            }
                        }/*End of TST_KeyGenType*/
                        
                        /*Preform the Sign operation */
                        /*---------------------------*/
                        switch(TST_HashType)
                        {
                            case CRYS_HASH_SHA1_mode:
                                TST_Status = CRYS_RSA_PSS_SHA1_Sign(&TST_ContextPrivate,
                                                                        &TST_PrivateKey,
                                                                        CRYS_PKCS1_MGF1,
                                                                        TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_SaltLength,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_input_hash_SHA1,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_output_signiture,
                                                                        &TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_KeySize);
                                break;
                            case CRYS_HASH_SHA224_mode:
                                TST_Status = CRYS_RSA_PSS_SHA224_Sign(&TST_ContextPrivate,
                                                                        &TST_PrivateKey,
                                                                        CRYS_PKCS1_MGF1,
                                                                        TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_SaltLength,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_input_hash_SHA1,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_output_signiture,
                                                                        &TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_KeySize);
                                break;
                            case CRYS_HASH_SHA256_mode:
                                TST_Status = CRYS_RSA_PSS_SHA256_Sign(&TST_ContextPrivate,
                                                                          &TST_PrivateKey,
                                                                          CRYS_PKCS1_MGF1,
                                                                          TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_SaltLength,
                                                                          TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_input_hash_SHA1,
                                                                          TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_output_signiture,
                                                                          &TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_KeySize);
                                break;
                            case CRYS_HASH_SHA384_mode:
                                #ifndef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
                                TST_Status = CRYS_RSA_PSS_SHA384_Sign(&TST_ContextPrivate,
                                                                        &TST_PrivateKey,
                                                                        CRYS_PKCS1_MGF1,
                                                                        TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_SaltLength,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_input_hash_SHA1,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_output_signiture,
                                                                        &TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_KeySize);
                                break;
                                #else
                                continue;
                                #endif
                            case CRYS_HASH_SHA512_mode:
                                #ifndef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
                                TST_Status = CRYS_RSA_PSS_SHA512_Sign(&TST_ContextPrivate,
                                                                        &TST_PrivateKey,
                                                                        CRYS_PKCS1_MGF1,
                                                                        TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_SaltLength,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_input_hash_SHA1,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_output_signiture,
                                                                        &TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_KeySize);
                                break;
                                #else
                                continue;
                                #endif
                            default:
                                break;
                        }                                                                    
                        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_PDSS_XXX_Sign",
                                                                TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_Name,
                                                                TST_MasterTestName);

                                                                                                                                                              
                        /*Print the outputted signature*/
                        ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                                "The outputted signature - \n",
                                                TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_output_signiture,
                                                TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_KeySize);

                        /*Preform the Verify operation */
                        /*-----------------------------*/ 
                        switch(TST_HashType)
                        {
                            case CRYS_HASH_SHA1_mode:
                                TST_Status = CRYS_RSA_PSS_SHA1_Verify(&TST_ContextPublic,
                                                                        &TST_PublicKey,
                                                                        CRYS_PKCS1_MGF1,
                                                                        TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_SaltLength,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_input_hash_SHA1,
                                                                        TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_output_signiture);
                                break;
                            case CRYS_HASH_SHA224_mode:
                                TST_Status = CRYS_RSA_PSS_SHA224_Verify(&TST_ContextPublic,
                                                                            &TST_PublicKey,
                                                                            CRYS_PKCS1_MGF1,
                                                                            TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_SaltLength,
                                                                            TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_input_hash_SHA1,
                                                                            TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_output_signiture);
                                break;
                            case CRYS_HASH_SHA256_mode:
                                TST_Status = CRYS_RSA_PSS_SHA256_Verify(&TST_ContextPublic,
                                                                            &TST_PublicKey,
                                                                            CRYS_PKCS1_MGF1,
                                                                            TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_SaltLength,
                                                                            TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_input_hash_SHA1,
                                                                            TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_output_signiture);
                                break;
                            case CRYS_HASH_SHA384_mode:
                                #ifndef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
                                TST_Status = CRYS_RSA_PSS_SHA384_Verify(&TST_ContextPublic,
                                                                            &TST_PublicKey,
                                                                            CRYS_PKCS1_MGF1,
                                                                            TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_SaltLength,
                                                                            TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_input_hash_SHA1,
                                                                            TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_output_signiture);
                                break;
                                #else
                                continue;
                                #endif
                            case CRYS_HASH_SHA512_mode:
                                #ifndef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
                                TST_Status = CRYS_RSA_PSS_SHA512_Verify(&TST_ContextPublic,
                                                                            &TST_PublicKey,
                                                                            CRYS_PKCS1_MGF1,
                                                                            TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_SaltLength,
                                                                            TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_input_hash_SHA1,
                                                                            TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_output_signiture);
                                break;
                                #else
                                continue;
                                #endif
                            default:
                                break;
                        }

                        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_PSS_XXX_Verify",
                                                                TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_Name,
                                                                TST_MasterTestName);

                    }/*End of TST_Index loop*/
                }/*End of TST_CRT_Mode loop*/
            }/*End of MD5 if*/
        }/*End of Hash Type*/
    }/*End of TST_KeyGenType loop*/ 
  return TST_Status; 
   
EXIT_ON_ERROR:
  {
   return DX_FAILURE;
  }
}/*End of CRYS_ATP_TST_RSA_Acceptance_PSS_ExternalHash_SignVerify*/

/*****************************************************/
/*TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************/
/*--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************/
/*TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************/
/*--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************/
/*****************************************************/
/***************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_RSA_Get_PrivAndPubKey      
 * 
 * Inputs:
 *   TST_KeyGeneration_TestStatus - A parameter defining if to run the test with internal key generation 
 *                                TST_INTERNAL_KG - Run the test with internal key generation and external keys 
 *                                TST_EXTERNAL_KEYPAIR  - Run the test with external key only - NO keyGen
 *
 * Outputs:
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function Is an Acceptance test for RSA Get Privet Key and Get Public Key                  
 *  This function Generate a RSA key and get its privet\public key and
 *  compare it to the test vector
 *
 *  Key sizes used     (in bits) - 512/1024/2048
 *  This also outputs the results to an output buffer or the same input buffer
 * Algorithm:       
 *  1. Prepare keys 
 *     a. Generate keys 
 *     or
 *     a. Build the Private Key from external exponent and modulus
 *     b. Build the Public Key from External exponent and modulus
 *  3. Call the CRYS_RSA_Get_Privet_key to preform the Get operation
 *  4. Call the CRYS_RSA_Get_Public_key to preform the Get operation
 *  5. Compare the outputted result to the expected one                  
 *
 **************************/
CRYSError_t CRYS_ATP_TST_RSA_Get_PrivAndPubKey(int TST_KeyGeneration_TestStatus)
{
    /*** TST Variables Definitions ****/
  TST_RSA_TestGetKeyDataStruct TST_RSA_Tmp_Key;
  
    int         TST_index,TST_CRT_Mode;
    CRYSError_t TST_Status=0;
    int         TST_KeyGenType;
    char        TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_RSA_Get_PrivAndPubKey"; 
    int       TST_HashType;
    CRYS_RSA_HASH_OpMode_t TST_RSAHashType=CRYS_RSA_HASH_NumOfModes;

    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"*********************** \n"));
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"** CRYS_ATP_TST_RSA_Get_Private_And_Public_Key Tests    ** \n"));      
    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"*********************** \n \n"));
    
    /****************/
    /***** Start of test ********/
    /****************/
    /* Setting if to use internal key generation */
    if (TST_KeyGeneration_TestStatus == TST_NO_INTERNAL_KEYGEN)
        TST_KeyGenType = TST_EXTERNAL_KEYPAIR;
    else
        TST_KeyGenType = TST_INTERNAL_KG;

    for ( ; TST_KeyGenType <2 ; TST_KeyGenType++)/*TST_KeyGenType = 0 Internal KeyGen // TST_KeyGenType = 1 External keys*/
    {   
        for (TST_HashType = 0 ; TST_HashType < CRYS_HASH_NumOfModes ; TST_HashType++)
        {   
            switch(TST_HashType)
            {
                case CRYS_HASH_SHA1_mode:
                    TST_RSAHashType = CRYS_RSA_HASH_SHA1_mode;
                    break;
                case CRYS_HASH_SHA224_mode:
                    TST_RSAHashType = CRYS_RSA_HASH_SHA224_mode;
                    break;
                case CRYS_HASH_SHA256_mode:
                    TST_RSAHashType = CRYS_RSA_HASH_SHA256_mode;
                    break;
                case CRYS_HASH_SHA384_mode:
                    #ifndef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
                    TST_RSAHashType = CRYS_RSA_HASH_SHA384_mode;
                    break;
                    #else
                    continue;
                    #endif
                case CRYS_HASH_SHA512_mode:
                    #ifndef LLF_HASH_SHA384_SHA512_NOT_SUPPORTED
                    TST_RSAHashType = CRYS_RSA_HASH_SHA512_mode;
                    break;
                    #else
                    continue;
                    #endif
                default:
                    break;
            }
            for (TST_CRT_Mode = 0 ; TST_CRT_Mode < 2 ; TST_CRT_Mode++)/* TST_CRT_Mode = 0 no CRT // TST_CRT_Mode = 1 CRT mode*/
            {   
                for (TST_index =0 ; TST_index < TST_NumberOf_RSA_Acceptance_ExportPrivAndPubKey ; TST_index++) /*To run on all test vectors*/
                {
                    /* CRT constraint CRT with 512 key should not work */
                    if ((TST_CRT_Mode == TST_CRT_MODE) && 
                        (TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_KeySize == TST_RSA_512_KEY_SIZE_IN_BYTES))
                    continue;
                    /* SHA384 & SHA512 are not used with RSA 512bit keys*/
                    if ((TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_KeySize == TST_RSA_512_KEY_SIZE_IN_BYTES)&&
                        ((TST_RSAHashType == CRYS_RSA_HASH_SHA384_mode)||(TST_RSAHashType == CRYS_RSA_HASH_SHA512_mode)))
                    continue;
                    
                    /*Cleaning the keys to prevent test errors with KG*/ 
                    DX_VOS_MemSet(&TST_PrivateKey,0,sizeof(TST_PrivateKey));
                    DX_VOS_MemSet(&TST_PublicKey,0,sizeof(TST_PublicKey));
                    
                    /* output buffer != input buffer    */
                    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"************************ \n"));
                    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," %s[test 5.%d]\n",TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_Name,TST_index)); 
                    CRYS_ATP_TST_PrintHASHType((CRYS_HASH_OperationMode_t)TST_HashType);                
                    if (TST_CRT_Mode == TST_NO_CRT)
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," Non CRT mode  \n"));
                    else
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," CRT mode  \n"));
                    if (TST_KeyGenType == TST_INTERNAL_KG)
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," Internal Key Generation  \n"));
                    else
                        ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," External Keys  \n"));                                
                    ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"************************ \n"));
            

                    /* Debug massages */
                    ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"Input Data Size [byte] = %1d \n" ,TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_input_dataSize));
      
                    if (TST_CRT_Mode == TST_NO_CRT)
                    {
                       /* Generate The private key */
                       /*****************************/
                       TST_Status = CRYS_RSA_Build_PrivKey(&TST_PrivateKey,
                                                              TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_PrivetExponent_D,
                                                              TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_KeySize,
                                                              TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_PublicExponent_E,
                                                              TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_PubExponentSize,                                               
                                                              TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_Modulus_N,
                                                              TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_KeySize);                                  
                                                      
                        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Build_PrivKey",
                                                   TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_Name,
                                                   TST_MasterTestName);

                         /* Build the public Key */
                         /*----------------------*/

                         TST_Status = CRYS_RSA_Build_PubKey(&TST_PublicKey,
                                                               TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_PublicExponent_E,
                                                               TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_PubExponentSize,
                                                               TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_Modulus_N,
                                                               TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_KeySize);
                                                          
                                                        
                        TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Build_PubKey",
                                                   TST_RSA_Acceptance_SignVerify_TestData[TST_index].TST_Name,
                                                   TST_MasterTestName);

                        TST_RSA_Tmp_Key.TST_PrivExponentSize=TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_KeySize;
                        TST_RSA_Tmp_Key.TST_PubExponentSize=TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_PubExponentSize;
                        TST_RSA_Tmp_Key.TST_ModulusSize=TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_KeySize;

                        TST_Status = CRYS_RSA_Get_PubKey(&TST_PublicKey,
                                                            TST_RSA_Tmp_Key.TST_PublicExponent_E,
                                                            &TST_RSA_Tmp_Key.TST_PubExponentSize,
                                                            TST_RSA_Tmp_Key.TST_Modulus_N,
                                                            &TST_RSA_Tmp_Key.TST_ModulusSize);
                                                                                   
                       TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Get_PubKey",
                                                  TST_RSA_Tmp_Key.TST_Name,
                                                  TST_MasterTestName);

                      /*Comparing The Public Key - Exponent */
                      TST_Status = (CRYSError_t)DX_VOS_MemCmp(TST_RSA_Tmp_Key.TST_PublicExponent_E,
                                                                 TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_PublicExponent_E,
                                                                 TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_PubExponentSize);

                      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Compare_Public_Key Exponent",
                                                 TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_Name,
                                                 TST_MasterTestName);


                     /*Comparing The Public Key - Modulus */
                     TST_Status = (CRYSError_t)DX_VOS_MemCmp(TST_RSA_Tmp_Key.TST_Modulus_N,
                                                                TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_Modulus_N,
                                                                TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_KeySize);

                     TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Compare_Public_Key Modulus",
                                                TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_Name,
                                                TST_MasterTestName);


                     TST_Status = CRYS_RSA_Get_PrivKey(&TST_PrivateKey,
                                                          TST_RSA_Tmp_Key.TST_PrivetExponent_D,
                                                          &TST_RSA_Tmp_Key.TST_PrivExponentSize,
                                                          TST_RSA_Tmp_Key.TST_PublicExponent_E,
                                                          &TST_RSA_Tmp_Key.TST_PubExponentSize,
                                                          TST_RSA_Tmp_Key.TST_Modulus_N,
                                                          &TST_RSA_Tmp_Key.TST_ModulusSize);


                    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Get_PrivKey",
                                               TST_RSA_Tmp_Key.TST_Name,
                                               TST_MasterTestName);

                   /*Comparing The Private Key */
                   TST_Status = (CRYSError_t)DX_VOS_MemCmp(TST_RSA_Tmp_Key.TST_PrivetExponent_D,
                                                              TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_PrivetExponent_D,
                                                              TST_RSA_Tmp_Key.TST_PrivExponentSize);

                   TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Compare_Private_Key",
                                              TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_Name,
                                              TST_MasterTestName);
                }
               else
               {
                 /* Get The private CRT key */
                 /***************************/
                 TST_Status = CRYS_RSA_Build_PrivKeyCRT(&TST_PrivateKey,
                                                           TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_P,
                                                           TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_DPSize,
                                                           TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_Q,
                                                           TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_DPSize,
                                                           TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_dP,
                                                           TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_DPSize,
                                                           TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_dQ,
                                                           TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_DPSize,
                                                           TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_Qinv,
                                                           TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_DPSize);

                TST_ATP_CHECK_ERROR_RETURN(TST_Status,"RYS_RSA_Build_PrivKeyCRT",
                                           TST_RSA_Acceptance_ExtHashSignVerify_TestData[TST_index].TST_Name,
                                           TST_MasterTestName);


                TST_RSA_Tmp_Key.TST_pSize    =  TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_DPSize;
                TST_RSA_Tmp_Key.TST_qSize    =  TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_DPSize;
                TST_RSA_Tmp_Key.TST_DpSize   =  TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_DPSize;
                TST_RSA_Tmp_Key.TST_DqSize   =  TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_DPSize;
                TST_RSA_Tmp_Key.TST_QinvSize =  TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_DPSize;

                TST_Status = CRYS_RSA_Get_PrivKeyCRT(&TST_PrivateKey,
                                                        TST_RSA_Tmp_Key.TST_P,
                                                        &TST_RSA_Tmp_Key.TST_pSize,
                                                        TST_RSA_Tmp_Key.TST_Q,
                                                        &TST_RSA_Tmp_Key.TST_qSize,
                                                        TST_RSA_Tmp_Key.TST_Dp,
                                                        &TST_RSA_Tmp_Key.TST_DpSize,
                                                        TST_RSA_Tmp_Key.TST_Dq,
                                                        &TST_RSA_Tmp_Key.TST_DqSize,
                                                        TST_RSA_Tmp_Key.TST_Qinv,
                                                        &TST_RSA_Tmp_Key.TST_QinvSize);

               TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Get_PrivKeyCRT",
                                          TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_Name,
                                          TST_MasterTestName);

               /*Comparing The Private Key */
                TST_Status = (CRYSError_t)DX_VOS_MemCmp(TST_RSA_Tmp_Key.TST_P,
                                                           TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_P,
                                                           TST_RSA_Tmp_Key.TST_pSize);
                
                TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Compare_CRT_Private_Key",
                                           TST_RSA_Acceptance_ExportPrivAndPubKey_TestData[TST_index].TST_Name,
                                           TST_MasterTestName);
              }
                }/*End of TST_Index loop*/
            }/*End of TST_CRT_Mode loop*/
         }/* End of TST_HashType loop*/
        }/* End of TST_KeyGenType loop*/
     return TST_Status; 

EXIT_ON_ERROR:

   return DX_FAILURE;

}/*End of CRYS_ATP_TST_RSA_Acceptance_PKCS1v15_Integrated_SignVerify*/
/*****************************************************/
/*TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************/
/*--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************/
/*TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************/
/*--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************/
/*****************************************************/
#endif
